<?php

namespace App\Models;

use Illuminate\Database\Eloquent\Model;
use Illuminate\Database\Eloquent\Relations\BelongsTo;

class Transaction extends Model
{
    protected $fillable = [
        'user_id',
        'loan_id',
        'type',
        'amount',
        'fee_amount',
        'tax_amount',
        'currency',
        'mpesa_transaction_id',
        'mpesa_phone_number',
        'reference_number',
        'from_user_id',
        'to_user_id',
        'from_account_id',
        'to_account_id',
        'payment_method',
        'transaction_category',
        'status',
        'mpesa_response',
        'notes',
        'description',
        'metadata',
        'processed_at',
    ];

    protected $casts = [
        'amount' => 'decimal:2',
        'fee_amount' => 'decimal:2',
        'tax_amount' => 'decimal:2',
        'mpesa_response' => 'array',
        'metadata' => 'array',
        'processed_at' => 'datetime',
    ];

    public function user(): BelongsTo
    {
        return $this->belongsTo(User::class);
    }

    public function fromUser(): BelongsTo
    {
        return $this->belongsTo(User::class, 'from_user_id');
    }

    public function toUser(): BelongsTo
    {
        return $this->belongsTo(User::class, 'to_user_id');
    }

    public function loan(): BelongsTo
    {
        return $this->belongsTo(Loan::class);
    }

    public function getTypeLabelAttribute()
    {
        return match($this->type) {
            'mpesa_deposit' => 'M-Pesa Deposit',
            'mpesa_withdrawal' => 'M-Pesa Withdrawal',
            'account_transfer' => 'Account Transfer',
            'loan_disbursement' => 'Loan Disbursement',
            'loan_repayment' => 'Loan Repayment',
            'savings_deposit' => 'Savings Deposit',
            'savings_withdrawal' => 'Savings Withdrawal',
            'registration_fee' => 'Registration Fee',
            'share_purchase' => 'Share Purchase',
            'dividend_payment' => 'Dividend Payment',
            'interest_payment' => 'Interest Payment',
            'fee_payment' => 'Fee Payment',
            'other' => 'Other Transaction',
            default => ucfirst(str_replace('_', ' ', $this->type))
        };
    }

    public function getTypeColorAttribute()
    {
        return match($this->type) {
            'mpesa_deposit', 'savings_deposit' => 'success',
            'mpesa_withdrawal', 'savings_withdrawal' => 'danger',
            'account_transfer' => 'info',
            'loan_disbursement', 'loan_repayment' => 'primary',
            'registration_fee', 'share_purchase', 'fee_payment' => 'warning',
            'dividend_payment', 'interest_payment' => 'success',
            default => 'secondary'
        };
    }

    public function getStatusColorAttribute()
    {
        return match($this->status) {
            'completed' => 'success',
            'pending' => 'warning',
            'failed' => 'danger',
            default => 'secondary'
        };
    }

    public function getFormattedAmountAttribute()
    {
        return $this->currency . ' ' . number_format($this->amount, 2);
    }

    public function getTotalAmountAttribute()
    {
        return $this->amount + $this->fee_amount + $this->tax_amount;
    }
}
