<?php

namespace App\Models;

use Illuminate\Database\Eloquent\Model;
use Illuminate\Database\Eloquent\Relations\BelongsTo;

class LoanCollateral extends Model
{
    protected $fillable = [
        'loan_guarantee_id',
        'item_type',
        'item_description',
        'estimated_value',
        'ownership_document_type',
        'ownership_document_number',
        'ownership_document_date',
        'ownership_details',
        'document_file_path',
        'status',
        'valuation_notes',
        'approved_at',
        'approved_by',
    ];

    protected $casts = [
        'estimated_value' => 'decimal:2',
        'ownership_document_date' => 'date',
        'approved_at' => 'datetime',
    ];

    public function loanGuarantee(): BelongsTo
    {
        return $this->belongsTo(LoanGuarantee::class);
    }

    public function approvedBy(): BelongsTo
    {
        return $this->belongsTo(User::class, 'approved_by');
    }

    public function getItemTypeLabelAttribute()
    {
        return match($this->item_type) {
            'vehicle' => 'Vehicle',
            'land' => 'Land/Property',
            'house' => 'House',
            'machinery' => 'Machinery/Equipment',
            'other' => 'Other',
            default => ucfirst($this->item_type)
        };
    }

    public function getDocumentTypeLabelAttribute()
    {
        return match($this->ownership_document_type) {
            'title_deed' => 'Title Deed',
            'log_book' => 'Log Book',
            'receipt' => 'Purchase Receipt',
            'certificate' => 'Ownership Certificate',
            'other' => 'Other Document',
            default => ucfirst(str_replace('_', ' ', $this->ownership_document_type))
        };
    }

    public function getStatusColorAttribute()
    {
        return match($this->status) {
            'approved' => 'success',
            'pending' => 'warning',
            'rejected' => 'danger',
            default => 'secondary'
        };
    }
}
