<?php

namespace App\Models;

use Illuminate\Database\Eloquent\Model;
use Illuminate\Support\Facades\Cache;

class Currency extends Model
{
    protected $fillable = [
        'name',
        'code',
        'symbol',
        'is_primary',
        'is_active',
        'sort_order',
    ];

    protected $casts = [
        'is_primary' => 'boolean',
        'is_active' => 'boolean',
        'sort_order' => 'integer',
    ];

    /**
     * Scope for active currencies
     */
    public function scopeActive($query)
    {
        return $query->where('is_active', true);
    }

    /**
     * Scope for ordering currencies
     */
    public function scopeOrdered($query)
    {
        return $query->orderBy('sort_order')->orderBy('name');
    }

    /**
     * Get the primary currency
     */
    public static function getPrimary()
    {
        return Cache::remember('primary_currency', 3600, function () {
            return static::where('is_primary', true)->first();
        });
    }

    /**
     * Get current system currency (primary or default)
     */
    public static function getCurrent()
    {
        $primary = static::getPrimary();
        if ($primary) {
            return $primary;
        }

        // Fallback to KES if no primary currency is set
        return static::where('code', 'KES')->first() ?? static::first();
    }

    /**
     * Set a currency as primary
     */
    public function setAsPrimary()
    {
        // Remove primary status from all currencies
        static::where('is_primary', true)->update(['is_primary' => false]);

        // Set this currency as primary
        $this->update(['is_primary' => true]);

        // Clear cache
        Cache::forget('primary_currency');

        return $this;
    }

    /**
     * Get formatted currency symbol
     */
    public function getFormattedSymbolAttribute()
    {
        return $this->symbol ?: $this->code;
    }

    /**
     * Format amount with currency
     */
    public function formatAmount($amount, $includeSymbol = true)
    {
        $formatted = number_format($amount, 2);

        if ($includeSymbol) {
            return $this->symbol . $formatted;
        }

        return $formatted;
    }

    /**
     * Boot the model
     */
    protected static function boot()
    {
        parent::boot();

        // Clear cache when currency is updated
        static::saved(function ($currency) {
            Cache::forget('primary_currency');
        });

        static::deleted(function ($currency) {
            Cache::forget('primary_currency');
        });
    }
}
