<?php

namespace App\Models;

use Illuminate\Database\Eloquent\Model;
use Illuminate\Database\Eloquent\Relations\BelongsTo;

class Communication extends Model
{
    protected $fillable = [
        'subject',
        'content',
        'recipient_type',
        'recipient_ids',
        'status',
        'sent_at',
        'scheduled_at',
        'priority',
        'created_by',
    ];

    protected $casts = [
        'recipient_ids' => 'array',
        'sent_at' => 'datetime',
        'scheduled_at' => 'datetime',
    ];

    /**
     * Get the user who created this communication
     */
    public function creator(): BelongsTo
    {
        return $this->belongsTo(User::class, 'created_by');
    }

    /**
     * Scope for draft communications
     */
    public function scopeDraft($query)
    {
        return $query->where('status', 'draft');
    }

    /**
     * Scope for sent communications
     */
    public function scopeSent($query)
    {
        return $query->where('status', 'sent');
    }

    /**
     * Scope for scheduled communications
     */
    public function scopeScheduled($query)
    {
        return $query->where('status', 'scheduled');
    }

    /**
     * Get recipients based on recipient type
     */
    public function getRecipients()
    {
        switch ($this->recipient_type) {
            case 'all_members':
                return User::whereHas('roles', function ($query) {
                    $query->where('name', 'Member');
                })->get();
            case 'individual':
                return User::whereIn('id', $this->recipient_ids ?? [])->get();
            case 'specific_groups':
                // For future implementation - could be based on membership status, branch, etc.
                return collect();
            default:
                return collect();
        }
    }
}
