<?php

namespace App\Http\Controllers;

use App\Models\User;
use App\Models\Branch;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\Mail;
use Maatwebsite\Excel\Facades\Excel;
use App\Exports\UsersExport;
use PDF;

class UserController extends Controller
{
    /**
     * Display a listing of the resource.
     */
    public function index()
    {
        $users = User::with('branch')->paginate(15);
        return view('adminlte.users.index', compact('users'));
    }

    /**
     * Show the form for creating a new resource.
     */
    public function create()
    {
        $branches = Branch::all();
        return view('adminlte.users.create', compact('branches'));
    }

    /**
     * Store a newly created resource in storage.
     */
    public function store(Request $request)
    {
        $request->validate([
            'first_name' => 'required|string|max:255',
            'second_name' => 'required|string|max:255',
            'phone_number' => 'required|string|max:255|unique:users',
            'id_number' => 'required|string|max:255|unique:users',
            'email' => 'required|string|email|max:255|unique:users',
            'password' => 'required|string|min:8',
            'role' => 'required|in:member,admin,manager',
            'branch_id' => 'nullable|exists:branches,id',
        ]);

        User::create([
            'first_name' => $request->first_name,
            'second_name' => $request->second_name,
            'name' => $request->first_name . ' ' . $request->second_name,
            'phone_number' => $request->phone_number,
            'id_number' => $request->id_number,
            'email' => $request->email,
            'password' => bcrypt($request->password),
            'role' => $request->role,
            'branch_id' => $request->branch_id,
        ]);

        return redirect()->route('users.index')->with('success', 'User created successfully.');
    }

    /**
     * Display the specified resource.
     */
    public function show(User $user)
    {
        return view('adminlte.users.show', compact('user'));
    }

    /**
     * Show the form for editing the specified resource.
     */
    public function edit(User $user)
    {
        $branches = Branch::all();
        return view('adminlte.users.edit', compact('user', 'branches'));
    }

    /**
     * Update the specified resource in storage.
     */
    public function update(Request $request, User $user)
    {
        $request->validate([
            'first_name' => 'required|string|max:255',
            'second_name' => 'required|string|max:255',
            'phone_number' => 'required|string|max:255|unique:users,phone_number,' . $user->id,
            'id_number' => 'required|string|max:255|unique:users,id_number,' . $user->id,
            'email' => 'required|string|email|max:255|unique:users,email,' . $user->id,
            'password' => 'nullable|string|min:8',
            'role' => 'required|in:member,admin,manager',
            'branch_id' => 'nullable|exists:branches,id',
        ]);

        $userData = [
            'first_name' => $request->first_name,
            'second_name' => $request->second_name,
            'name' => $request->first_name . ' ' . $request->second_name,
            'phone_number' => $request->phone_number,
            'id_number' => $request->id_number,
            'email' => $request->email,
            'role' => $request->role,
            'branch_id' => $request->branch_id,
        ];

        // Only update password if provided
        if ($request->filled('password')) {
            $userData['password'] = bcrypt($request->password);
        }

        $user->update($userData);

        return redirect()->route('users.index')->with('success', 'User updated successfully.');
    }

    /**
     * Remove the specified resource from storage.
     */
    public function destroy(User $user)
    {
        $user->delete();
        return redirect()->route('users.index')->with('success', 'User deleted successfully.');
    }

    /**
     * Verify user account.
     */
    public function verify(User $user)
    {
        if ($user->email_verified_at) {
            return redirect()->back()->with('info', 'User is already verified.');
        }

        $user->update([
            'email_verified_at' => now(),
        ]);

        // Update related records
        $updates = [];

        // Update membership status to active if it exists
        if ($user->membership && $user->membership->status !== 'active') {
            $user->membership->update(['status' => 'active']);
            $updates[] = 'Membership activated';
        }

        // Log verification activity (optional)
        // You could add activity logging here if needed

        $message = 'User verified successfully.';
        if (!empty($updates)) {
            $message .= ' Updated: ' . implode(', ', $updates);
        }

        return redirect()->back()->with('success', $message);
    }

    /**
     * Transfer user to another branch.
     */
    public function transfer(Request $request, User $user)
    {
        $request->validate([
            'branch_id' => 'required|exists:branches,id',
        ]);

        $user->update(['branch_id' => $request->branch_id]);

        return redirect()->back()->with('success', 'User transferred successfully.');
    }

    /**
     * Send custom message to user.
     */
    public function sendMessage(Request $request, User $user)
    {
        $request->validate([
            'message' => 'required|string',
            'type' => 'required|in:email,whatsapp',
        ]);

        if ($request->type === 'email') {
            Mail::raw($request->message, function ($mail) use ($user) {
                $mail->to($user->email)->subject('Message from Morio Sacco');
            });
        } elseif ($request->type === 'whatsapp') {
            // Implement WhatsApp sending logic here
            // For now, just log or placeholder
        }

        return redirect()->back()->with('success', 'Message sent successfully.');
    }

    /**
     * Export users to PDF.
     */
    public function exportPdf()
    {
        $users = User::with('branch')->get();

        $pdf = PDF::loadView('adminlte.users.pdf', compact('users'));

        return $pdf->download('users_report_' . date('Y-m-d') . '.pdf');
    }

    /**
     * Export users to Excel.
     */
    public function exportExcel()
    {
        return Excel::download(new UsersExport, 'users_report_' . date('Y-m-d') . '.xlsx');
    }

    /**
     * Export users to CSV.
     */
    public function exportCsv()
    {
        return Excel::download(new UsersExport, 'users_report_' . date('Y-m-d') . '.csv');
    }
}
