<?php

namespace App\Http\Controllers;

use Illuminate\Http\Request;
use App\Models\SystemSetting;
use Illuminate\Support\Facades\Storage;

class SystemSettingController extends Controller
{
    /**
     * Display a listing of the resource.
     */
    public function index()
    {
        // Get all settings grouped by category
        $settings = SystemSetting::orderBy('category')->orderBy('key')->get()->groupBy('category');

        // Define default settings structure
        $categories = [
            'general' => [
                'title' => 'General Settings',
                'icon' => 'fas fa-cogs',
                'settings' => [
                    'system_name' => ['label' => 'System Name', 'type' => 'string', 'default' => 'Morio Sacco'],
                    'system_description' => ['label' => 'System Description', 'type' => 'text', 'default' => 'Modern Sacco Management System'],
                    'system_footer' => ['label' => 'System Footer', 'type' => 'text', 'default' => 'Powered by <a href="https://zillahtechnologies.co.ke" target="_blank">Zillah Technologies LTD</a> | Morio Sacco'],
                    'timezone' => ['label' => 'Timezone', 'type' => 'string', 'default' => 'Africa/Nairobi'],
                ]
            ],
            'email' => [
                'title' => 'Email Settings',
                'icon' => 'fas fa-envelope',
                'settings' => [
                    'smtp_host' => ['label' => 'SMTP Host', 'type' => 'string', 'default' => 'smtp.gmail.com'],
                    'smtp_port' => ['label' => 'SMTP Port', 'type' => 'number', 'default' => '587'],
                    'smtp_username' => ['label' => 'SMTP Username', 'type' => 'string', 'default' => ''],
                    'smtp_password' => ['label' => 'SMTP Password', 'type' => 'string', 'default' => ''],
                    'smtp_encryption' => ['label' => 'SMTP Encryption', 'type' => 'string', 'default' => 'tls'],
                    'from_email' => ['label' => 'From Email', 'type' => 'string', 'default' => 'noreply@morio-sacco.com'],
                    'from_name' => ['label' => 'From Name', 'type' => 'string', 'default' => 'Morio Sacco'],
                ]
            ],
            'social' => [
                'title' => 'Social Media Links',
                'icon' => 'fas fa-share-alt',
                'settings' => [
                    'facebook_url' => ['label' => 'Facebook URL', 'type' => 'string', 'default' => ''],
                    'twitter_url' => ['label' => 'Twitter URL', 'type' => 'string', 'default' => ''],
                    'instagram_url' => ['label' => 'Instagram URL', 'type' => 'string', 'default' => ''],
                    'linkedin_url' => ['label' => 'LinkedIn URL', 'type' => 'string', 'default' => ''],
                    'youtube_url' => ['label' => 'YouTube URL', 'type' => 'string', 'default' => ''],
                ]
            ],
            'seo' => [
                'title' => 'SEO Settings',
                'icon' => 'fas fa-search',
                'settings' => [
                    'meta_title' => ['label' => 'Meta Title', 'type' => 'string', 'default' => 'Morio Sacco - Modern Sacco Management'],
                    'meta_description' => ['label' => 'Meta Description', 'type' => 'text', 'default' => 'Manage your sacco operations efficiently with our modern management system.'],
                    'meta_keywords' => ['label' => 'Meta Keywords', 'type' => 'string', 'default' => 'sacco, savings, loans, management'],
                    'google_analytics_id' => ['label' => 'Google Analytics ID', 'type' => 'string', 'default' => ''],
                ]
            ],
            'contact' => [
                'title' => 'Contact Information',
                'icon' => 'fas fa-address-book',
                'settings' => [
                    'contact_phone' => ['label' => 'Phone Number', 'type' => 'string', 'default' => '+254 700 000 000'],
                    'contact_email' => ['label' => 'Email Address', 'type' => 'string', 'default' => 'info@morio-sacco.co.ke'],
                    'contact_address' => ['label' => 'Office Address', 'type' => 'text', 'default' => 'Koinange Street, Nairobi, Kenya'],
                    'app_url' => ['label' => 'App URL', 'type' => 'string', 'default' => 'https://morio-sacco.co.ke'],
                ]
            ],
            'system' => [
                'title' => 'System Assets',
                'icon' => 'fas fa-image',
                'settings' => [
                    'logo_path' => ['label' => 'System Logo', 'type' => 'file', 'default' => 'images/morio-sacco-logo.png'],
                    'favicon_path' => ['label' => 'Favicon', 'type' => 'file', 'default' => 'favicon.ico'],
                ]
            ],
            'database' => [
                'title' => 'Database Management',
                'icon' => 'fas fa-database',
                'settings' => [
                    // This will be handled specially in the view
                ]
            ]
        ];

        return view('adminlte.system-settings.index', compact('settings', 'categories'));
    }

    /**
     * Update the specified resource in storage.
     */
    public function update(Request $request)
    {
        $validated = $request->validate([
            'settings' => 'required|array',
            'settings.*' => 'nullable|string',
        ]);

        foreach ($validated['settings'] as $key => $value) {
            // Handle file uploads
            if ($request->hasFile("files.{$key}")) {
                $file = $request->file("files.{$key}");
                $path = $file->store('system', 'public');
                $value = $path;
            }

            SystemSetting::updateOrCreate(
                ['key' => $key],
                [
                    'value' => $value,
                    'type' => $this->getSettingType($key),
                    'category' => $this->getSettingCategory($key),
                    'label' => $this->getSettingLabel($key),
                    'description' => $this->getSettingDescription($key),
                    'is_public' => $this->isPublicSetting($key),
                ]
            );
        }

        // Clear cache
        SystemSetting::clearCache();

        return redirect()->back()->with('success', 'System settings updated successfully!');
    }

    /**
     * Get setting type
     */
    private function getSettingType($key)
    {
        $types = [
            'system_description' => 'text',
            'system_footer' => 'text',
            'meta_description' => 'text',
            'contact_address' => 'text',
            'smtp_port' => 'number',
            'logo_path' => 'file',
            'favicon_path' => 'file',
        ];

        return $types[$key] ?? 'string';
    }

    /**
     * Get setting category
     */
    private function getSettingCategory($key)
    {
        $categories = [
            'system_name' => 'general',
            'system_description' => 'general',
            'system_footer' => 'general',
            'timezone' => 'general',
            'smtp_host' => 'email',
            'smtp_port' => 'email',
            'smtp_username' => 'email',
            'smtp_password' => 'email',
            'smtp_encryption' => 'email',
            'from_email' => 'email',
            'from_name' => 'email',
            'facebook_url' => 'social',
            'twitter_url' => 'social',
            'instagram_url' => 'social',
            'linkedin_url' => 'social',
            'youtube_url' => 'social',
            'meta_title' => 'seo',
            'meta_description' => 'seo',
            'meta_keywords' => 'seo',
            'google_analytics_id' => 'seo',
            'contact_phone' => 'contact',
            'contact_email' => 'contact',
            'contact_address' => 'contact',
            'app_url' => 'contact',
            'logo_path' => 'system',
            'favicon_path' => 'system',
        ];

        return $categories[$key] ?? 'general';
    }

    /**
     * Get setting label
     */
    private function getSettingLabel($key)
    {
        $labels = [
            'system_name' => 'System Name',
            'system_description' => 'System Description',
            'system_footer' => 'System Footer',
            'timezone' => 'Timezone',
            'smtp_host' => 'SMTP Host',
            'smtp_port' => 'SMTP Port',
            'smtp_username' => 'SMTP Username',
            'smtp_password' => 'SMTP Password',
            'smtp_encryption' => 'SMTP Encryption',
            'from_email' => 'From Email',
            'from_name' => 'From Name',
            'facebook_url' => 'Facebook URL',
            'twitter_url' => 'Twitter URL',
            'instagram_url' => 'Instagram URL',
            'linkedin_url' => 'LinkedIn URL',
            'youtube_url' => 'YouTube URL',
            'meta_title' => 'Meta Title',
            'meta_description' => 'Meta Description',
            'meta_keywords' => 'Meta Keywords',
            'google_analytics_id' => 'Google Analytics ID',
            'contact_phone' => 'Phone Number',
            'contact_email' => 'Email Address',
            'contact_address' => 'Office Address',
            'app_url' => 'App URL',
            'logo_path' => 'System Logo',
            'favicon_path' => 'Favicon',
        ];

        return $labels[$key] ?? ucfirst(str_replace('_', ' ', $key));
    }

    /**
     * Get setting description
     */
    private function getSettingDescription($key)
    {
        $descriptions = [
            'system_name' => 'The name of your sacco system',
            'system_description' => 'Brief description of your sacco system',
            'system_footer' => 'Footer text displayed at the bottom of all pages',
            'timezone' => 'System timezone',
            'smtp_host' => 'SMTP server hostname',
            'smtp_port' => 'SMTP server port (usually 587 for TLS)',
            'smtp_username' => 'SMTP authentication username',
            'smtp_password' => 'SMTP authentication password',
            'smtp_encryption' => 'SMTP encryption type (tls/ssl)',
            'from_email' => 'Default sender email address',
            'from_name' => 'Default sender name',
            'facebook_url' => 'Your Facebook page URL',
            'twitter_url' => 'Your Twitter profile URL',
            'instagram_url' => 'Your Instagram profile URL',
            'linkedin_url' => 'Your LinkedIn page URL',
            'youtube_url' => 'Your YouTube channel URL',
            'meta_title' => 'SEO title for homepage',
            'meta_description' => 'SEO description for homepage',
            'meta_keywords' => 'SEO keywords (comma-separated)',
            'google_analytics_id' => 'Google Analytics tracking ID',
            'contact_phone' => 'Primary contact phone number',
            'contact_email' => 'Primary contact email address',
            'contact_address' => 'Physical office address',
            'app_url' => 'Application website URL',
            'logo_path' => 'Path to system logo file',
            'favicon_path' => 'Path to favicon file',
        ];

        return $descriptions[$key] ?? '';
    }

    /**
     * Check if setting is public
     */
    private function isPublicSetting($key)
    {
        $publicSettings = [
            'system_name',
            'system_description',
            'system_footer',
            'contact_phone',
            'contact_email',
            'contact_address',
            'app_url',
            'facebook_url',
            'twitter_url',
            'instagram_url',
            'linkedin_url',
            'youtube_url',
            'meta_title',
            'meta_description',
            'meta_keywords',
            'google_analytics_id',
            'logo_path',
            'favicon_path',
        ];

        return in_array($key, $publicSettings);
    }

    /**
     * Export database backup
     */
    public function exportDatabase()
    {
        try {
            $filename = 'morio_sacco_backup_' . date('Y-m-d_H-i-s') . '.sql';
            $path = storage_path('app/backups/' . $filename);

            // Ensure backup directory exists
            $backupDir = dirname($path);
            if (!file_exists($backupDir)) {
                mkdir($backupDir, 0755, true);
            }

            // Get database connection details
            $dbHost = env('DB_HOST', '127.0.0.1');
            $dbName = env('DB_DATABASE');
            $dbUser = env('DB_USERNAME');
            $dbPass = env('DB_PASSWORD');

            // Try mysqldump first (for Linux/Unix systems)
            $command = "mysqldump --user={$dbUser} --password={$dbPass} --host={$dbHost} {$dbName} > \"{$path}\"";

            $returnVar = null;
            $output = null;
            exec($command, $output, $returnVar);

            if ($returnVar !== 0) {
                // If mysqldump fails, try alternative approach using PHP
                $this->createBackupUsingPHP($path);
            }

            if (!file_exists($path) || filesize($path) === 0) {
                return redirect()->back()->with('error', 'Failed to create database backup.');
            }

            return response()->download($path)->deleteFileAfterSend(true);

        } catch (\Exception $e) {
            return redirect()->back()->with('error', 'Error creating backup: ' . $e->getMessage());
        }
    }

    /**
     * Create backup using PHP when mysqldump is not available
     */
    private function createBackupUsingPHP($path)
    {
        $pdo = \DB::connection()->getPdo();
        $tables = $pdo->query("SHOW TABLES")->fetchAll(\PDO::FETCH_COLUMN);

        $sql = "-- Morio Sacco Database Backup\n";
        $sql .= "-- Generated on " . date('Y-m-d H:i:s') . "\n\n";
        $sql .= "SET FOREIGN_KEY_CHECKS = 0;\n\n";

        foreach ($tables as $table) {
            // Get table structure
            $stmt = $pdo->query("SHOW CREATE TABLE `{$table}`");
            $createTable = $stmt->fetch(\PDO::FETCH_ASSOC);
            $sql .= $createTable['Create Table'] . ";\n\n";

            // Get table data
            $stmt = $pdo->query("SELECT * FROM `{$table}`");
            $rows = $stmt->fetchAll(\PDO::FETCH_ASSOC);

            if (!empty($rows)) {
                $columns = array_keys($rows[0]);
                $sql .= "INSERT INTO `{$table}` (`" . implode('`, `', $columns) . "`) VALUES\n";

                $values = [];
                foreach ($rows as $row) {
                    $rowValues = [];
                    foreach ($row as $value) {
                        if ($value === null) {
                            $rowValues[] = 'NULL';
                        } else {
                            $rowValues[] = $pdo->quote($value);
                        }
                    }
                    $values[] = "(" . implode(', ', $rowValues) . ")";
                }

                $sql .= implode(",\n", $values) . ";\n\n";
            }
        }

        $sql .= "SET FOREIGN_KEY_CHECKS = 1;\n";

        file_put_contents($path, $sql);
    }

    /**
     * Import database backup
     */
    public function importDatabase(Request $request)
    {
        $request->validate([
            'backup_file' => 'required|file|mimes:sql|max:51200', // 50MB max
        ]);

        try {
            $file = $request->file('backup_file');
            $path = $file->getRealPath();

            // Get database connection details
            $dbHost = env('DB_HOST', '127.0.0.1');
            $dbName = env('DB_DATABASE');
            $dbUser = env('DB_USERNAME');
            $dbPass = env('DB_PASSWORD');

            // Try mysql command first
            $command = "mysql --user={$dbUser} --password={$dbPass} --host={$dbHost} {$dbName} < \"{$path}\"";

            $returnVar = null;
            $output = null;
            exec($command, $output, $returnVar);

            if ($returnVar !== 0) {
                // If mysql command fails, try PHP-based import
                $this->restoreBackupUsingPHP($path);
            }

            return redirect()->back()->with('success', 'Database backup restored successfully!');

        } catch (\Exception $e) {
            return redirect()->back()->with('error', 'Error restoring backup: ' . $e->getMessage());
        }
    }

    /**
     * Restore backup using PHP when mysql command is not available
     */
    private function restoreBackupUsingPHP($path)
    {
        $pdo = \DB::connection()->getPdo();

        // Read the SQL file
        $sql = file_get_contents($path);

        // Split into individual statements
        $statements = array_filter(array_map('trim', explode(';', $sql)));

        foreach ($statements as $statement) {
            if (!empty($statement) && !preg_match('/^--/', $statement)) {
                try {
                    $pdo->exec($statement);
                } catch (\Exception $e) {
                    // Log error but continue with other statements
                    \Log::error('SQL Import Error: ' . $e->getMessage() . ' - Statement: ' . substr($statement, 0, 100));
                }
            }
        }
    }
}
