<?php

namespace App\Http\Controllers;

use App\Models\User;
use App\Models\Membership;
use App\Models\Saving;
use App\Models\Loan;
use App\Models\Transaction;
use App\Models\Share;
use App\Models\Dividend;
use Illuminate\Http\Request;
use Barryvdh\DomPDF\Facade\Pdf;
use Maatwebsite\Excel\Facades\Excel;
use App\Exports\FinancialReportExport;
use App\Exports\MemberActivityReportExport;
use App\Exports\TransactionSummaryReportExport;
use Carbon\Carbon;

class ReportController extends Controller
{
    /**
     * Display the reports dashboard
     */
    public function index()
    {
        // Real-time dashboard data
        $dashboardData = $this->getDashboardData();

        return view('adminlte.reports.index', compact('dashboardData'));
    }

    /**
     * Financial Summary Report
     */
    public function financialSummary(Request $request)
    {
        $startDate = $request->get('start_date', Carbon::now()->startOfMonth());
        $endDate = $request->get('end_date', Carbon::now()->endOfMonth());

        $data = $this->getFinancialSummaryData($startDate, $endDate);

        if ($request->has('export')) {
            return $this->exportFinancialReport($data, $request->get('format', 'pdf'));
        }

        return view('adminlte.reports.financial-summary', compact('data', 'startDate', 'endDate'));
    }

    /**
     * Member Activity Report
     */
    public function memberActivity(Request $request)
    {
        $startDate = $request->get('start_date', Carbon::now()->startOfMonth());
        $endDate = $request->get('end_date', Carbon::now()->endOfMonth());

        $data = $this->getMemberActivityData($startDate, $endDate);

        if ($request->has('export')) {
            return $this->exportMemberActivityReport($data, $request->get('format', 'pdf'));
        }

        return view('adminlte.reports.member-activity', compact('data', 'startDate', 'endDate'));
    }

    /**
     * Transaction Summary Report
     */
    public function transactionSummary(Request $request)
    {
        $startDate = $request->get('start_date', Carbon::now()->startOfMonth());
        $endDate = $request->get('end_date', Carbon::now()->endOfMonth());

        $data = $this->getTransactionSummaryData($startDate, $endDate);

        if ($request->has('export')) {
            return $this->exportTransactionSummaryReport($data, $request->get('format', 'pdf'));
        }

        return view('adminlte.reports.transaction-summary', compact('data', 'startDate', 'endDate'));
    }

    /**
     * Loan Performance Report
     */
    public function loanPerformance(Request $request)
    {
        $data = $this->getLoanPerformanceData();

        if ($request->has('export')) {
            return $this->exportLoanPerformanceReport($data, $request->get('format', 'pdf'));
        }

        return view('adminlte.reports.loan-performance', compact('data'));
    }

    /**
     * Savings Performance Report
     */
    public function savingsPerformance(Request $request)
    {
        $data = $this->getSavingsPerformanceData();

        if ($request->has('export')) {
            return $this->exportSavingsPerformanceReport($data, $request->get('format', 'pdf'));
        }

        return view('adminlte.reports.savings-performance', compact('data'));
    }

    /**
     * Get real-time dashboard data
     */
    private function getDashboardData()
    {
        return [
            'total_members' => Membership::count(),
            'active_members' => Membership::where('status', 'active')->count(),
            'total_savings' => Saving::sum('balance'),
            'total_loans' => Loan::where('status', 'active')->sum('amount'),
            'monthly_transactions' => Transaction::whereMonth('created_at', Carbon::now()->month)->count(),
            'pending_loans' => Loan::where('status', 'pending')->count(),
            'total_shares' => Share::sum('amount'),
            'monthly_deposits' => Transaction::where('type', 'savings_deposit')
                ->whereMonth('created_at', Carbon::now()->month)
                ->sum('amount'),
            'monthly_withdrawals' => Transaction::where('type', 'savings_withdrawal')
                ->whereMonth('created_at', Carbon::now()->month)
                ->sum('amount'),
            'monthly_registration_fees' => Transaction::where('type', 'registration_fee')
                ->whereMonth('created_at', Carbon::now()->month)
                ->sum('amount'),
        ];
    }

    /**
     * Get financial summary data
     */
    private function getFinancialSummaryData($startDate, $endDate)
    {
        return [
            'income' => [
                'savings_deposits' => Transaction::where('type', 'savings_deposit')
                    ->whereBetween('created_at', [$startDate, $endDate])
                    ->sum('amount'),
                'loan_repayments' => Transaction::where('type', 'loan_repayment')
                    ->whereBetween('created_at', [$startDate, $endDate])
                    ->sum('amount'),
                'registration_fees' => Transaction::where('type', 'registration_fee')
                    ->whereBetween('created_at', [$startDate, $endDate])
                    ->sum('amount'),
                'share_purchases' => Transaction::where('type', 'share_purchase')
                    ->whereBetween('created_at', [$startDate, $endDate])
                    ->sum('amount'),
            ],
            'expenses' => [
                'loan_disbursements' => Transaction::where('type', 'loan_disbursement')
                    ->whereBetween('created_at', [$startDate, $endDate])
                    ->sum('amount'),
                'savings_withdrawals' => Transaction::where('type', 'savings_withdrawal')
                    ->whereBetween('created_at', [$startDate, $endDate])
                    ->sum('amount'),
                'dividend_payments' => Transaction::where('type', 'dividend_payment')
                    ->whereBetween('created_at', [$startDate, $endDate])
                    ->sum('amount'),
                'fee_payments' => Transaction::where('type', 'fee_payment')
                    ->whereBetween('created_at', [$startDate, $endDate])
                    ->sum('amount'),
            ],
            'balances' => [
                'total_savings' => Saving::sum('balance'),
                'total_loans_outstanding' => Loan::whereIn('status', ['active', 'approved'])->sum('amount'),
                'total_shares' => Share::sum('amount'),
            ],
            'period_summary' => [
                'net_income' => 0, // Will be calculated
                'total_transactions' => Transaction::whereBetween('created_at', [$startDate, $endDate])->count(),
                'active_loans' => Loan::where('status', 'active')->count(),
                'active_savings_accounts' => Saving::where('status', 'active')->count(),
            ],
        ];
    }

    /**
     * Get member activity data
     */
    private function getMemberActivityData($startDate, $endDate)
    {
        $members = Membership::with(['user', 'user.savings', 'user.loans', 'user.transactions' => function($query) use ($startDate, $endDate) {
            $query->whereBetween('created_at', [$startDate, $endDate]);
        }])->get();

        $memberData = [];
        foreach ($members as $membership) {
            $user = $membership->user;
            $memberData[] = [
                'member' => $membership,
                'savings_balance' => $user->savings->sum('balance'),
                'loan_balance' => $user->loans->whereIn('status', ['active', 'approved'])->sum('amount'),
                'share_balance' => $user->shares->sum('amount') ?? 0,
                'transaction_count' => $user->transactions->count(),
                'last_activity' => $user->transactions->max('created_at'),
                'total_deposits' => $user->transactions->where('type', 'savings_deposit')->sum('amount'),
                'total_withdrawals' => $user->transactions->where('type', 'savings_withdrawal')->sum('amount'),
            ];
        }

        return $memberData;
    }

    /**
     * Get transaction summary data
     */
    private function getTransactionSummaryData($startDate, $endDate)
    {
        $transactions = Transaction::with(['user'])
            ->whereBetween('created_at', [$startDate, $endDate])
            ->orderBy('created_at', 'desc')
            ->get();

        $summary = [
            'by_type' => $transactions->groupBy('type')->map(function($group) {
                return [
                    'count' => $group->count(),
                    'total_amount' => $group->sum('amount'),
                    'total_fees' => $group->sum('fee_amount'),
                ];
            }),
            'by_status' => $transactions->groupBy('status')->map(function($group) {
                return [
                    'count' => $group->count(),
                    'total_amount' => $group->sum('amount'),
                ];
            }),
            'by_payment_method' => $transactions->whereNotNull('payment_method')
                ->groupBy('payment_method')
                ->map(function($group) {
                    return [
                        'count' => $group->count(),
                        'total_amount' => $group->sum('amount'),
                    ];
                }),
            'daily_totals' => $transactions->groupBy(function($transaction) {
                return $transaction->created_at->format('Y-m-d');
            })->map(function($dayTransactions) {
                return [
                    'count' => $dayTransactions->count(),
                    'total_amount' => $dayTransactions->sum('amount'),
                    'deposits' => $dayTransactions->whereIn('type', ['savings_deposit', 'mpesa_deposit'])->sum('amount'),
                    'withdrawals' => $dayTransactions->whereIn('type', ['savings_withdrawal', 'mpesa_withdrawal'])->sum('amount'),
                ];
            }),
        ];

        return $summary;
    }

    /**
     * Get loan performance data
     */
    private function getLoanPerformanceData()
    {
        $loans = Loan::with(['user'])->get();

        return [
            'by_status' => $loans->groupBy('status')->map(function($group) {
                return [
                    'count' => $group->count(),
                    'total_amount' => $group->sum('amount'),
                    'average_amount' => $group->avg('amount'),
                ];
            }),
            'by_term' => $loans->groupBy('term_months')->map(function($group) {
                return [
                    'count' => $group->count(),
                    'total_amount' => $group->sum('amount'),
                ];
            }),
            'repayment_performance' => [
                'total_disbursed' => $loans->where('status', 'approved')->sum('amount'),
                'total_repaid' => Transaction::where('type', 'loan_repayment')->sum('amount'),
                'outstanding_balance' => $loans->whereIn('status', ['active', 'approved'])->sum('amount') - Transaction::where('type', 'loan_repayment')->sum('amount'),
            ],
            'default_risk' => [
                'overdue_loans' => $loans->where('status', 'active')
                    ->where('created_at', '<', Carbon::now()->subMonths(6))
                    ->count(),
                'high_value_loans' => $loans->where('amount', '>', 100000)->count(),
            ],
        ];
    }

    /**
     * Get savings performance data
     */
    private function getSavingsPerformanceData()
    {
        $savings = Saving::with(['user'])->get();

        return [
            'by_type' => $savings->groupBy('type')->map(function($group) {
                return [
                    'count' => $group->count(),
                    'total_balance' => $group->sum('balance'),
                    'average_balance' => $group->avg('balance'),
                ];
            }),
            'by_status' => $savings->groupBy('status')->map(function($group) {
                return [
                    'count' => $group->count(),
                    'total_balance' => $group->sum('balance'),
                ];
            }),
            'growth_metrics' => [
                'total_accounts' => $savings->count(),
                'active_accounts' => $savings->where('status', 'active')->count(),
                'total_balance' => $savings->sum('balance'),
                'average_balance' => $savings->avg('balance'),
                'monthly_deposits' => Transaction::where('type', 'savings_deposit')
                    ->whereMonth('created_at', Carbon::now()->month)
                    ->sum('amount'),
                'monthly_withdrawals' => Transaction::where('type', 'savings_withdrawal')
                    ->whereMonth('created_at', Carbon::now()->month)
                    ->sum('amount'),
            ],
            'member_distribution' => $savings->groupBy(function($saving) {
                $balance = $saving->balance;
                if ($balance < 1000) return 'Under 1K';
                if ($balance < 5000) return '1K - 5K';
                if ($balance < 10000) return '5K - 10K';
                if ($balance < 50000) return '10K - 50K';
                return 'Over 50K';
            })->map(function($group) {
                return [
                    'count' => $group->count(),
                    'total_balance' => $group->sum('balance'),
                ];
            }),
        ];
    }

    /**
     * Export methods for different report types
     */
    private function exportFinancialReport($data, $format)
    {
        if ($format === 'excel') {
            return Excel::download(new FinancialReportExport($data), 'financial-summary-' . date('Y-m-d') . '.xlsx');
        }

        $pdf = Pdf::loadView('adminlte.reports.exports.financial-summary-pdf', compact('data'));
        return $pdf->download('financial-summary-' . date('Y-m-d') . '.pdf');
    }

    private function exportMemberActivityReport($data, $format)
    {
        if ($format === 'excel') {
            return Excel::download(new MemberActivityReportExport($data), 'member-activity-' . date('Y-m-d') . '.xlsx');
        }

        $pdf = Pdf::loadView('adminlte.reports.exports.member-activity-pdf', compact('data'));
        return $pdf->download('member-activity-' . date('Y-m-d') . '.pdf');
    }

    private function exportTransactionSummaryReport($data, $format)
    {
        if ($format === 'excel') {
            return Excel::download(new TransactionSummaryReportExport($data), 'transaction-summary-' . date('Y-m-d') . '.xlsx');
        }

        $pdf = Pdf::loadView('adminlte.reports.exports.transaction-summary-pdf', compact('data'));
        return $pdf->download('transaction-summary-' . date('Y-m-d') . '.pdf');
    }

    private function exportLoanPerformanceReport($data, $format)
    {
        if ($format === 'excel') {
            return Excel::download(new \App\Exports\LoanPerformanceReportExport($data), 'loan-performance-' . date('Y-m-d') . '.xlsx');
        }

        $pdf = Pdf::loadView('adminlte.reports.exports.loan-performance-pdf', compact('data'));
        return $pdf->download('loan-performance-' . date('Y-m-d') . '.pdf');
    }

    private function exportSavingsPerformanceReport($data, $format)
    {
        if ($format === 'excel') {
            return Excel::download(new \App\Exports\SavingsPerformanceReportExport($data), 'savings-performance-' . date('Y-m-d') . '.xlsx');
        }

        $pdf = Pdf::loadView('adminlte.reports.exports.savings-performance-pdf', compact('data'));
        return $pdf->download('savings-performance-' . date('Y-m-d') . '.pdf');
    }
}
