<?php

namespace App\Http\Controllers;

use App\Models\MoneyMarketAccount;
use App\Models\Membership;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\DB;
use Carbon\Carbon;
use Maatwebsite\Excel\Facades\Excel;
use Barryvdh\DomPDF\Facade\Pdf;

class MoneyMarketAccountController extends Controller
{
    /**
     * Display a listing of the resource.
     */
    public function index(Request $request)
    {
        $query = MoneyMarketAccount::with('membership.user');

        // Filter by status
        if ($request->has('status') && $request->status !== '') {
            $query->where('status', $request->status);
        }

        // Filter by membership
        if ($request->has('membership_id') && $request->membership_id !== '') {
            $query->where('membership_id', $request->membership_id);
        }

        // Search by account number or member name
        if ($request->has('search') && $request->search !== '') {
            $search = $request->search;
            $query->where(function($q) use ($search) {
                $q->where('account_number', 'like', "%{$search}%")
                  ->orWhereHas('membership.user', function($userQuery) use ($search) {
                      $userQuery->where('first_name', 'like', "%{$search}%")
                               ->orWhere('second_name', 'like', "%{$search}%")
                               ->orWhere('full_name', 'like', "%{$search}%");
                  });
            });
        }

        $moneyMarketAccounts = $query->orderBy('created_at', 'desc')->paginate(15);

        // Get statistics
        $stats = [
            'total_accounts' => MoneyMarketAccount::count(),
            'active_accounts' => MoneyMarketAccount::where('status', 'active')->count(),
            'matured_accounts' => MoneyMarketAccount::where('status', 'matured')->count(),
            'total_principal' => MoneyMarketAccount::sum('principal_amount'),
            'total_interest_earned' => MoneyMarketAccount::sum('interest_earned'),
            'nearing_maturity' => MoneyMarketAccount::nearingMaturity()->count(),
        ];

        $memberships = Membership::with('user')->get();

        return view('adminlte.money-market-accounts.index', compact('moneyMarketAccounts', 'stats', 'memberships'));
    }

    /**
     * Show the form for creating a new resource.
     */
    public function create()
    {
        $memberships = Membership::with('user')->where('status', 'active')->get();
        return view('adminlte.money-market-accounts.create', compact('memberships'));
    }

    /**
     * Store a newly created resource in storage.
     */
    public function store(Request $request)
    {
        $validated = $request->validate([
            'membership_id' => 'required|exists:memberships,id',
            'principal_amount' => 'required|numeric|min:1000',
            'interest_rate' => 'required|numeric|min:0|max:50',
            'interest_type' => 'required|in:simple,compound',
            'compounding_frequency' => 'required_if:interest_type,compound|in:monthly,quarterly,annually',
            'term_months' => 'required|integer|min:1|max:120',
            'start_date' => 'required|date|after_or_equal:today',
            'currency_code' => 'required|string|size:3',
            'notes' => 'nullable|string|max:500',
        ]);

        DB::transaction(function () use ($validated) {
            $startDate = Carbon::parse($validated['start_date']);
            $maturityDate = $startDate->copy()->addMonths($validated['term_months']);

            $account = MoneyMarketAccount::create([
                'membership_id' => $validated['membership_id'],
                'account_number' => MoneyMarketAccount::generateAccountNumber(),
                'principal_amount' => $validated['principal_amount'],
                'interest_rate' => $validated['interest_rate'],
                'interest_type' => $validated['interest_type'],
                'compounding_frequency' => $validated['interest_type'] === 'compound' ? $validated['compounding_frequency'] : null,
                'term_months' => $validated['term_months'],
                'start_date' => $startDate,
                'maturity_date' => $maturityDate,
                'currency_code' => $validated['currency_code'],
                'notes' => $validated['notes'],
                'total_amount' => $validated['principal_amount'], // Will be updated with interest
            ]);

            // Calculate initial interest
            $account->updateInterestEarned();

            // Log the account creation transaction
            $account->logTransaction('account_creation', $validated['principal_amount'], 'Money market account created');
        });

        return redirect()->route('money-market-accounts.index')
                        ->with('success', 'Money market account created successfully!');
    }

    /**
     * Display the specified resource.
     */
    public function show(MoneyMarketAccount $moneyMarketAccount)
    {
        $moneyMarketAccount->load('membership.user');
        return view('adminlte.money-market-accounts.show', compact('moneyMarketAccount'));
    }

    /**
     * Show the form for editing the specified resource.
     */
    public function edit(MoneyMarketAccount $moneyMarketAccount)
    {
        if ($moneyMarketAccount->status !== 'active') {
            return redirect()->route('money-market-accounts.index')
                           ->with('error', 'Only active accounts can be edited.');
        }

        $memberships = Membership::with('user')->where('status', 'active')->get();
        return view('adminlte.money-market-accounts.edit', compact('moneyMarketAccount', 'memberships'));
    }

    /**
     * Update the specified resource in storage.
     */
    public function update(Request $request, MoneyMarketAccount $moneyMarketAccount)
    {
        if ($moneyMarketAccount->status !== 'active') {
            return redirect()->route('money-market-accounts.index')
                           ->with('error', 'Only active accounts can be updated.');
        }

        $validated = $request->validate([
            'interest_rate' => 'required|numeric|min:0|max:50',
            'notes' => 'nullable|string|max:500',
        ]);

        $moneyMarketAccount->update($validated);

        return redirect()->route('money-market-accounts.index')
                        ->with('success', 'Money market account updated successfully!');
    }

    /**
     * Withdraw from the money market account
     */
    public function withdraw(Request $request, MoneyMarketAccount $moneyMarketAccount)
    {
        $validated = $request->validate([
            'withdrawal_amount' => 'required|numeric|min:0',
        ]);

        if (!$moneyMarketAccount->withdraw($validated['withdrawal_amount'])) {
            return redirect()->back()
                           ->with('error', 'Withdrawal failed. Please check the amount and account status.');
        }

        return redirect()->route('money-market-accounts.index')
                        ->with('success', 'Withdrawal processed successfully!');
    }

    /**
     * Mark account as matured
     */
    public function markMatured(MoneyMarketAccount $moneyMarketAccount)
    {
        $moneyMarketAccount->markAsMatured();

        return redirect()->back()
                        ->with('success', 'Account marked as matured!');
    }

    /**
     * Update interest for all accounts
     */
    public function updateInterest()
    {
        $accounts = MoneyMarketAccount::whereIn('status', ['active', 'matured'])->get();
        $updated = 0;

        foreach ($accounts as $account) {
            $account->updateInterestEarned();
            $updated++;
        }

        return redirect()->back()
                        ->with('success', "Interest updated for {$updated} accounts!");
    }

    /**
     * Export money market accounts to PDF
     */
    public function exportPdf()
    {
        $moneyMarketAccounts = MoneyMarketAccount::with('membership.user')->get();
        $appUrl = config('app.url');

        $pdf = Pdf::loadView('adminlte.money-market-accounts.exports.pdf', compact('moneyMarketAccounts', 'appUrl'));

        return $pdf->download('money_market_accounts_' . date('Y-m-d_H-i-s') . '.pdf');
    }

    /**
     * Export money market accounts to Excel
     */
    public function exportExcel()
    {
        return Excel::download(new \App\Exports\MoneyMarketAccountsExport, 'money_market_accounts_' . date('Y-m-d_H-i-s') . '.xlsx');
    }

    /**
     * Export money market accounts to CSV
     */
    public function exportCsv()
    {
        return Excel::download(new \App\Exports\MoneyMarketAccountsExport, 'money_market_accounts_' . date('Y-m-d_H-i-s') . '.csv', \Maatwebsite\Excel\Excel::CSV);
    }

    /**
     * Remove the specified resource from storage.
     */
    public function destroy(MoneyMarketAccount $moneyMarketAccount)
    {
        if ($moneyMarketAccount->status !== 'cancelled') {
            return redirect()->route('money-market-accounts.index')
                           ->with('error', 'Only cancelled accounts can be deleted.');
        }

        $moneyMarketAccount->delete();

        return redirect()->route('money-market-accounts.index')
                        ->with('success', 'Money market account deleted successfully!');
    }
}
