<?php

namespace App\Http\Controllers;

use App\Models\Membership;
use App\Models\User;
use Illuminate\Http\Request;
use Maatwebsite\Excel\Facades\Excel;
use Barryvdh\DomPDF\Facade\Pdf;

class MembershipController extends Controller
{
    /**
     * Display a listing of the resource.
     */
    public function index(Request $request)
    {
        $query = Membership::with('user.branch');

        // Filter by status if provided
        if ($request->has('status') && in_array($request->status, ['active', 'inactive', 'suspended'])) {
            $query->where('status', $request->status);
        }

        $memberships = $query->paginate(15);

        // Calculate dynamic statistics from member behavior with caching
        $totalMembers = \Cache::remember('total_members_count', 300, function () {
            return Membership::count();
        });

        // Members with savings (active savings accounts) - optimized and cached
        $membersWithSavings = \Cache::remember('members_with_savings_count', 300, function () {
            return \DB::table('savings')
                ->where('status', 'active')
                ->distinct('user_id')
                ->count('user_id');
        });

        // Members with loans (any loan status) - optimized and cached
        $membersWithLoans = \Cache::remember('members_with_loans_count', 300, function () {
            return \DB::table('loans')
                ->distinct('user_id')
                ->count('user_id');
        });

        return view('adminlte.memberships.index', compact(
            'memberships',
            'totalMembers',
            'membersWithSavings',
            'membersWithLoans'
        ));
    }

    /**
     * Show the form for creating a new resource.
     */
    public function create()
    {
        $branches = \App\Models\Branch::all();

        return view('adminlte.memberships.create', compact('branches'));
    }

    /**
     * Store a newly created resource in storage.
     */
    public function store(Request $request)
    {
        $request->validate([
            'first_name' => 'required|string|max:255',
            'second_name' => 'required|string|max:255',
            'gender' => 'required|in:male,female',
            'occupation' => 'nullable|string|max:255',
            'custom_occupation' => 'nullable|string|max:255',
            'custom_occupation_value' => 'nullable|string|max:255',
            'id_number' => 'required|string|unique:users,id_number',
            'phone_number' => 'required|string|unique:users,phone_number',
            'email' => 'required|email|unique:users,email',
            'role' => 'required|in:member,admin,manager',
            'branch_id' => 'required|exists:branches,id',
            'membership_number' => 'required|string|unique:memberships,membership_number',
            'joining_date' => 'required|date',
            'status' => 'required|in:active,inactive,suspended',
            'share_capital' => 'required|numeric|min:0',
            'savings_balance' => 'required|numeric|min:0',
            'registration_fee' => 'required|numeric|min:0',
            'registration_fee_status' => 'required|in:pending,paid,waived',
            'passport_photo' => 'nullable|image|mimes:jpeg,png,jpg|max:2048',
            'notes' => 'nullable|string',
        ]);

        // Auto-generate password for members
        $password = $this->generateRandomPassword();

        // Create user first
        $user = User::create([
            'name' => $request->first_name . ' ' . $request->second_name,
            'first_name' => $request->first_name,
            'second_name' => $request->second_name,
            'phone_number' => $request->phone_number,
            'id_number' => $request->id_number,
            'email' => $request->email,
            'password' => bcrypt($password),
            'role' => $request->role,
            'branch_id' => $request->branch_id,
        ]);

        // Automatically assign Member role for member registrations
        if ($request->role === 'member') {
            $user->assignRole('Member');
        }

        // Handle passport photo upload
        $photoName = null;
        if ($request->hasFile('passport_photo')) {
            $photoName = time() . '_' . $user->id . '.' . $request->passport_photo->extension();
            $request->passport_photo->storeAs('memberships', $photoName, 'public');
        }

        // Handle occupation
        $occupation = $request->occupation;
        if ($occupation === 'other' && $request->filled('custom_occupation')) {
            $occupation = $request->custom_occupation;
        } elseif ($occupation === 'other' && $request->filled('custom_occupation_value')) {
            $occupation = $request->custom_occupation_value;
        }

        // Create membership
        Membership::create([
            'user_id' => $user->id,
            'gender' => $request->gender,
            'occupation' => $occupation,
            'membership_number' => $request->membership_number,
            'joining_date' => $request->joining_date,
            'status' => $request->status,
            'share_capital' => $request->share_capital,
            'savings_balance' => $request->savings_balance,
            'registration_fee' => $request->registration_fee,
            'registration_fee_status' => $request->registration_fee_status,
            'passport_photo' => $photoName,
            'notes' => $request->notes,
        ]);

        // Send login credentials email
        try {
            \Mail::to($user->email)->send(new \App\Mail\MemberLoginCredentials($user, $password));
        } catch (\Exception $e) {
            // Log the error but don't fail the registration
            \Log::error('Failed to send login credentials email: ' . $e->getMessage());
        }

        return redirect()->route('memberships.index')
                        ->with('success', 'Member registered successfully! Login credentials have been sent to their email.');
    }

    /**
     * Display the specified resource.
     */
    public function show(string $id)
    {
        $membership = Membership::with('user.branch')->findOrFail($id);

        // Fetch recent savings transactions
        $savingsTransactions = \App\Models\SavingsTransaction::with(['account'])
            ->whereHas('account', function($query) use ($membership) {
                $query->where('user_id', $membership->user_id);
            })
            ->latest('transaction_date')
            ->take(10)
            ->get()
            ->map(function($transaction) {
                return [
                    'date' => $transaction->transaction_date,
                    'type' => 'savings_' . $transaction->type,
                    'amount' => $transaction->amount,
                    'status' => 'completed',
                    'description' => $transaction->type_label . ' - ' . $transaction->account->type_label,
                    'module' => 'savings'
                ];
            });

        // Fetch recent loan activities
        $loanActivities = \App\Models\Loan::where('user_id', $membership->user_id)
            ->latest('created_at')
            ->take(10)
            ->get()
            ->map(function($loan) {
                return [
                    'date' => $loan->created_at,
                    'type' => 'loan_application',
                    'amount' => $loan->amount,
                    'status' => $loan->status,
                    'description' => 'Loan Application - KSh ' . number_format($loan->amount, 0),
                    'module' => 'loans'
                ];
            });

        // Fetch recent registration fee transactions
        $registrationFeeTransactions = \App\Models\Transaction::where('user_id', $membership->user_id)
            ->where('type', 'registration_fee')
            ->latest('created_at')
            ->take(5)
            ->get()
            ->map(function($transaction) {
                return [
                    'date' => $transaction->created_at,
                    'type' => 'registration_fee',
                    'amount' => $transaction->amount,
                    'status' => $transaction->status,
                    'description' => 'Registration Fee Payment - ' . ($transaction->payment_method === 'cash' ? 'Cash' : 'M-Pesa'),
                    'module' => 'membership'
                ];
            });

        // Combine and sort by date
        $recentActivities = $savingsTransactions->concat($loanActivities)->concat($registrationFeeTransactions)
            ->sortByDesc('date')
            ->take(10);

        return view('adminlte.memberships.show', compact('membership', 'recentActivities'));
    }

    /**
     * Show the form for editing the specified resource.
     */
    public function edit(string $id)
    {
        $membership = Membership::with('user.branch')->findOrFail($id);
        $branches = \App\Models\Branch::all();

        return view('adminlte.memberships.edit', compact('membership', 'branches'));
    }

    /**
     * Update the specified resource in storage.
     */
    public function update(Request $request, string $id)
    {
        $membership = Membership::findOrFail($id);

        $request->validate([
            'first_name' => 'required|string|max:255',
            'second_name' => 'required|string|max:255',
            'gender' => 'required|in:male,female',
            'occupation' => 'nullable|string|max:255',
            'custom_occupation' => 'nullable|string|max:255',
            'custom_occupation_value' => 'nullable|string|max:255',
            'phone_number' => 'required|string|unique:users,phone_number,' . $membership->user_id,
            'email' => 'required|email|unique:users,email,' . $membership->user_id,
            'password' => 'nullable|string|min:8',
            'role' => 'required|in:member,admin,manager',
            'branch_id' => 'required|exists:branches,id',
            'membership_number' => 'required|string|unique:memberships,membership_number,' . $id,
            'joining_date' => 'required|date',
            'status' => 'required|in:active,inactive,suspended',
            'share_capital' => 'required|numeric|min:0',
            'savings_balance' => 'required|numeric|min:0',
            'registration_fee' => 'required|numeric|min:0',
            'registration_fee_status' => 'required|in:pending,paid,waived',
            'passport_photo' => 'nullable|image|mimes:jpeg,png,jpg|max:2048',
            'notes' => 'nullable|string',
        ]);

        // Handle passport photo upload
        if ($request->hasFile('passport_photo')) {
            // Delete old photo if exists
            if ($membership->passport_photo && \Storage::disk('public')->exists('memberships/' . $membership->passport_photo)) {
                \Storage::disk('public')->delete('memberships/' . $membership->passport_photo);
            }

            $photoName = time() . '_' . $membership->id . '.' . $request->passport_photo->extension();
            $request->passport_photo->storeAs('memberships', $photoName, 'public');
            $membership->passport_photo = $photoName;
        }

        // Update user
        $userData = [
            'name' => $request->first_name . ' ' . $request->second_name,
            'first_name' => $request->first_name,
            'second_name' => $request->second_name,
            'phone_number' => $request->phone_number,
            'email' => $request->email,
            'role' => $request->role,
            'branch_id' => $request->branch_id,
        ];

        // Only update password if provided
        if ($request->filled('password')) {
            $userData['password'] = bcrypt($request->password);
        }

        $membership->user->update($userData);

        // Handle role assignment
        if ($request->role === 'member') {
            if (!$membership->user->hasRole('Member')) {
                $membership->user->assignRole('Member');
            }
        } else {
            if ($membership->user->hasRole('Member')) {
                $membership->user->removeRole('Member');
            }
        }

        // Handle occupation
        $occupation = $request->occupation;
        if ($occupation === 'other' && $request->filled('custom_occupation')) {
            $occupation = $request->custom_occupation;
        } elseif ($occupation === 'other' && $request->filled('custom_occupation_value')) {
            $occupation = $request->custom_occupation_value;
        }

        // Update membership
        $membership->update([
            'gender' => $request->gender,
            'occupation' => $occupation,
            'membership_number' => $request->membership_number,
            'joining_date' => $request->joining_date,
            'status' => $request->status,
            'share_capital' => $request->share_capital,
            'savings_balance' => $request->savings_balance,
            'registration_fee' => $request->registration_fee,
            'registration_fee_status' => $request->registration_fee_status,
            'notes' => $request->notes,
        ]);

        return redirect()->route('memberships.index')
                        ->with('success', 'Member updated successfully!');
    }

    /**
     * Remove the specified resource from storage.
     */
    public function destroy(string $id)
    {
        $membership = Membership::findOrFail($id);

        // Delete passport photo if exists
        if ($membership->passport_photo && \Storage::disk('public')->exists('memberships/' . $membership->passport_photo)) {
            \Storage::disk('public')->delete('memberships/' . $membership->passport_photo);
        }

        // Delete membership (this will also delete the associated user due to cascade)
        $membership->delete();

        return redirect()->route('memberships.index')
                        ->with('success', 'Member deleted successfully!');
    }

    public function payRegistrationFee(Request $request, Membership $membership)
    {
        $request->validate([
            'amount' => 'required|numeric|min:1',
            'mpesa_phone_number' => 'required|string|regex:/^254[0-9]{9}$/'
        ]);

        try {
            // Get M-Pesa credentials
            $mpesaConfig = \App\Models\MpesaCredential::where('is_active', true)->first();

            if (!$mpesaConfig) {
                return response()->json([
                    'success' => false,
                    'message' => 'M-Pesa configuration not found. Please contact administrator.'
                ]);
            }

            // Use the provided M-Pesa phone number
            $mpesaPhoneNumber = $request->mpesa_phone_number;

            // Initialize M-Pesa STK Push
            $mpesa = new \Safaricom\Mpesa\Mpesa();

            $response = $mpesa->stkpush([
                'BusinessShortCode' => $mpesaConfig->shortcode,
                'Password' => $this->generatePassword($mpesaConfig),
                'Timestamp' => date('YmdHis'),
                'TransactionType' => 'CustomerPayBillOnline',
                'Amount' => $request->amount,
                'PartyA' => $mpesaPhoneNumber,
                'PartyB' => $mpesaConfig->shortcode,
                'PhoneNumber' => $mpesaPhoneNumber,
                'CallBackURL' => route('mpesa.registration-fee.callback'),
                'AccountReference' => 'REG-FEE-' . $membership->membership_number,
                'TransactionDesc' => 'Registration Fee Payment'
            ]);

            // Create transaction record with the custom phone number
            \App\Models\Transaction::create([
                'user_id' => $membership->user_id,
                'type' => 'registration_fee',
                'amount' => $request->amount,
                'currency' => 'KES',
                'mpesa_transaction_id' => $response['MerchantRequestID'] ?? null,
                'mpesa_phone_number' => $mpesaPhoneNumber,
                'reference_number' => 'REG-FEE-' . $membership->membership_number,
                'payment_method' => 'mpesa_stk',
                'transaction_category' => 'membership_fee',
                'status' => 'pending',
                'mpesa_response' => json_encode($response),
                'description' => 'Registration fee payment for membership ' . $membership->membership_number,
                'notes' => 'Registration fee payment initiated via M-Pesa STK push to ' . $mpesaPhoneNumber,
                'metadata' => [
                    'membership_number' => $membership->membership_number,
                    'member_name' => $membership->user->full_name,
                    'payment_type' => 'registration_fee'
                ]
            ]);

            return response()->json([
                'success' => true,
                'message' => 'M-Pesa STK push initiated successfully to ' . $mpesaPhoneNumber,
                'data' => $response
            ]);

        } catch (\Exception $e) {
            return response()->json([
                'success' => false,
                'message' => 'Failed to initiate payment: ' . $e->getMessage()
            ]);
        }
    }

    public function payCashRegistrationFee(Request $request, Membership $membership)
    {
        $request->validate([
            'amount' => 'required|numeric|min:1',
            'notes' => 'nullable|string|max:255'
        ]);

        // Check if registration fee is already paid
        if ($membership->registration_fee_status === 'paid') {
            return response()->json([
                'success' => false,
                'message' => 'Registration fee has already been paid.'
            ]);
        }

        // Check if amount matches the required fee
        if ($request->amount != $membership->registration_fee) {
            return response()->json([
                'success' => false,
                'message' => 'Payment amount must match the registration fee of KSh ' . number_format($membership->registration_fee, 2)
            ]);
        }

        try {
            // Update membership status
            $membership->update([
                'registration_fee_status' => 'paid',
                'status' => 'active'
            ]);

            // Verify the user if not already verified
            if (!$membership->user->email_verified_at) {
                $membership->user->update([
                    'email_verified_at' => now()
                ]);
            }

            // Assign Member role if not already assigned
            if (!$membership->user->hasRole('Member')) {
                $membership->user->assignRole('Member');
            }

            // Create transaction record for cash payment
            \App\Models\Transaction::create([
                'user_id' => $membership->user_id,
                'type' => 'registration_fee',
                'amount' => $request->amount,
                'currency' => 'KES',
                'reference_number' => 'CASH-' . $membership->membership_number . '-' . date('YmdHis'),
                'payment_method' => 'cash',
                'transaction_category' => 'membership_fee',
                'status' => 'completed',
                'description' => 'Cash payment for registration fee - membership ' . $membership->membership_number,
                'notes' => 'Cash payment for registration fee' . ($request->notes ? ': ' . $request->notes : ''),
                'processed_at' => now(),
                'metadata' => [
                    'membership_number' => $membership->membership_number,
                    'member_name' => $membership->user->full_name,
                    'payment_type' => 'registration_fee',
                    'cashier_notes' => $request->notes ?? null
                ]
            ]);

            return response()->json([
                'success' => true,
                'message' => 'Cash payment recorded successfully. Member account has been activated.',
                'data' => [
                    'membership_number' => $membership->membership_number,
                    'member_name' => $membership->user->full_name,
                    'amount_paid' => $request->amount
                ]
            ]);

        } catch (\Exception $e) {
            return response()->json([
                'success' => false,
                'message' => 'Failed to process cash payment: ' . $e->getMessage()
            ]);
        }
    }

    private function generatePassword($mpesaConfig)
    {
        $timestamp = date('YmdHis');
        $password = base64_encode($mpesaConfig->shortcode . $mpesaConfig->passkey . $timestamp);
        return $password;
    }

    /**
     * Export memberships to PDF
     */
    public function exportPdf()
    {
        $memberships = Membership::with('user.branch')->get();

        $pdf = Pdf::loadView('adminlte.memberships.exports.pdf', compact('memberships'))
                  ->setPaper('a4', 'landscape');

        return $pdf->download('memberships_report_' . date('Y-m-d_H-i-s') . '.pdf');
    }

    /**
     * Export memberships to Excel
     */
    public function exportExcel()
    {
        return Excel::download(new \App\Exports\MembershipsExport, 'memberships_report_' . date('Y-m-d_H-i-s') . '.xlsx');
    }

    /**
     * Export member activities to CSV
     */
    public function exportMemberActivitiesCsv(Membership $membership)
    {
        // Fetch all savings transactions
        $savingsTransactions = \App\Models\SavingsTransaction::with(['account'])
            ->whereHas('account', function($query) use ($membership) {
                $query->where('user_id', $membership->user_id);
            })
            ->latest('transaction_date')
            ->get()
            ->map(function($transaction) {
                return [
                    'date' => $transaction->transaction_date->format('Y-m-d H:i:s'),
                    'type' => 'Savings ' . $transaction->type_label,
                    'amount' => $transaction->amount,
                    'status' => 'Completed',
                    'description' => $transaction->type_label . ' - ' . $transaction->account->type_label,
                    'module' => 'Savings'
                ];
            });

        // Fetch all loan activities
        $loanActivities = \App\Models\Loan::where('user_id', $membership->user_id)
            ->latest('created_at')
            ->get()
            ->map(function($loan) {
                return [
                    'date' => $loan->created_at->format('Y-m-d H:i:s'),
                    'type' => 'Loan Application',
                    'amount' => $loan->amount,
                    'status' => ucfirst($loan->status),
                    'description' => 'Loan Application - KSh ' . number_format($loan->amount, 0),
                    'module' => 'Loans'
                ];
            });

        // Fetch all registration fee transactions
        $registrationFeeTransactions = \App\Models\Transaction::where('user_id', $membership->user_id)
            ->where('type', 'registration_fee')
            ->latest('created_at')
            ->get()
            ->map(function($transaction) {
                return [
                    'date' => $transaction->created_at->format('Y-m-d H:i:s'),
                    'type' => 'Registration Fee',
                    'amount' => $transaction->amount,
                    'status' => ucfirst($transaction->status),
                    'description' => 'Registration Fee Payment - ' . ($transaction->payment_method === 'cash' ? 'Cash' : 'M-Pesa'),
                    'module' => 'Membership'
                ];
            });

        // Fetch all other financial transactions
        $otherTransactions = \App\Models\Transaction::where('user_id', $membership->user_id)
            ->whereNotIn('type', ['registration_fee'])
            ->latest('created_at')
            ->get()
            ->map(function($transaction) {
                return [
                    'date' => $transaction->created_at->format('Y-m-d H:i:s'),
                    'type' => $transaction->type_label,
                    'amount' => $transaction->amount,
                    'status' => ucfirst($transaction->status),
                    'description' => $transaction->description ?: 'Transaction processed',
                    'module' => 'Transactions'
                ];
            });

        // Combine all activities and sort by date
        $activities = $savingsTransactions
            ->concat($loanActivities)
            ->concat($registrationFeeTransactions)
            ->concat($otherTransactions)
            ->sortByDesc('date')
            ->values();

        $filename = 'member_' . $membership->membership_number . '_activities_' . date('Y-m-d_H-i-s') . '.csv';

        $headers = [
            'Content-Type' => 'text/csv',
            'Content-Disposition' => 'attachment; filename="' . $filename . '"',
        ];

        $callback = function() use ($activities) {
            $file = fopen('php://output', 'w');

            // CSV headers
            fputcsv($file, ['Date', 'Type', 'Amount', 'Status', 'Description', 'Module']);

            // CSV data
            foreach ($activities as $activity) {
                fputcsv($file, [
                    $activity['date'],
                    $activity['type'],
                    $activity['amount'],
                    $activity['status'],
                    $activity['description'],
                    $activity['module']
                ]);
            }

            fclose($file);
        };

        return response()->stream($callback, 200, $headers);
    }

    /**
     * Export member activities to Excel
     */
    public function exportMemberActivitiesExcel(Membership $membership)
    {
        // Fetch all savings transactions
        $savingsTransactions = \App\Models\SavingsTransaction::with(['account'])
            ->whereHas('account', function($query) use ($membership) {
                $query->where('user_id', $membership->user_id);
            })
            ->latest('transaction_date')
            ->get()
            ->map(function($transaction) {
                return [
                    'Date' => $transaction->transaction_date->format('Y-m-d H:i:s'),
                    'Type' => 'Savings ' . $transaction->type_label,
                    'Amount' => $transaction->amount,
                    'Status' => 'Completed',
                    'Description' => $transaction->type_label . ' - ' . $transaction->account->type_label,
                    'Module' => 'Savings'
                ];
            });

        // Fetch all loan activities
        $loanActivities = \App\Models\Loan::where('user_id', $membership->user_id)
            ->latest('created_at')
            ->get()
            ->map(function($loan) {
                return [
                    'Date' => $loan->created_at->format('Y-m-d H:i:s'),
                    'Type' => 'Loan Application',
                    'Amount' => $loan->amount,
                    'Status' => ucfirst($loan->status),
                    'Description' => 'Loan Application - KSh ' . number_format($loan->amount, 0),
                    'Module' => 'Loans'
                ];
            });

        // Fetch all registration fee transactions
        $registrationFeeTransactions = \App\Models\Transaction::where('user_id', $membership->user_id)
            ->where('type', 'registration_fee')
            ->latest('created_at')
            ->get()
            ->map(function($transaction) {
                return [
                    'Date' => $transaction->created_at->format('Y-m-d H:i:s'),
                    'Type' => 'Registration Fee',
                    'Amount' => $transaction->amount,
                    'Status' => ucfirst($transaction->status),
                    'Description' => 'Registration Fee Payment - ' . ($transaction->payment_method === 'cash' ? 'Cash' : 'M-Pesa'),
                    'Module' => 'Membership'
                ];
            });

        // Fetch all other financial transactions
        $otherTransactions = \App\Models\Transaction::where('user_id', $membership->user_id)
            ->whereNotIn('type', ['registration_fee'])
            ->latest('created_at')
            ->get()
            ->map(function($transaction) {
                return [
                    'Date' => $transaction->created_at->format('Y-m-d H:i:s'),
                    'Type' => $transaction->type_label,
                    'Amount' => $transaction->amount,
                    'Status' => ucfirst($transaction->status),
                    'Description' => $transaction->description ?: 'Transaction processed',
                    'Module' => 'Transactions'
                ];
            });

        // Combine all activities and sort by date
        $activities = $savingsTransactions
            ->concat($loanActivities)
            ->concat($registrationFeeTransactions)
            ->concat($otherTransactions)
            ->sortByDesc('Date')
            ->values();

        $filename = 'member_' . $membership->membership_number . '_activities_' . date('Y-m-d_H-i-s') . '.xlsx';

        return Excel::download(new class($activities) implements \Maatwebsite\Excel\Concerns\FromCollection, \Maatwebsite\Excel\Concerns\WithHeadings {
            private $activities;

            public function __construct($activities)
            {
                $this->activities = $activities;
            }

            public function collection()
            {
                return collect($this->activities);
            }

            public function headings(): array
            {
                return ['Date', 'Type', 'Amount', 'Status', 'Description', 'Module'];
            }
        }, $filename);
    }

    /**
     * Export member activities to PDF
     */
    public function exportMemberActivitiesPdf(Membership $membership)
    {
        // Fetch all savings transactions
        $savingsTransactions = \App\Models\SavingsTransaction::with(['account'])
            ->whereHas('account', function($query) use ($membership) {
                $query->where('user_id', $membership->user_id);
            })
            ->latest('transaction_date')
            ->get()
            ->map(function($transaction) {
                return [
                    'date' => $transaction->transaction_date->format('Y-m-d H:i:s'),
                    'type' => 'Savings ' . $transaction->type_label,
                    'amount' => $transaction->amount,
                    'status' => 'Completed',
                    'description' => $transaction->type_label . ' - ' . $transaction->account->type_label,
                    'module' => 'Savings'
                ];
            });

        // Fetch all loan activities
        $loanActivities = \App\Models\Loan::where('user_id', $membership->user_id)
            ->latest('created_at')
            ->get()
            ->map(function($loan) {
                return [
                    'date' => $loan->created_at->format('Y-m-d H:i:s'),
                    'type' => 'Loan Application',
                    'amount' => $loan->amount,
                    'status' => ucfirst($loan->status),
                    'description' => 'Loan Application - KSh ' . number_format($loan->amount, 0),
                    'module' => 'Loans'
                ];
            });

        // Fetch all registration fee transactions
        $registrationFeeTransactions = \App\Models\Transaction::where('user_id', $membership->user_id)
            ->where('type', 'registration_fee')
            ->latest('created_at')
            ->get()
            ->map(function($transaction) {
                return [
                    'date' => $transaction->created_at->format('Y-m-d H:i:s'),
                    'type' => 'Registration Fee',
                    'amount' => $transaction->amount,
                    'status' => ucfirst($transaction->status),
                    'description' => 'Registration Fee Payment - ' . ($transaction->payment_method === 'cash' ? 'Cash' : 'M-Pesa'),
                    'module' => 'Membership'
                ];
            });

        // Fetch all other financial transactions
        $otherTransactions = \App\Models\Transaction::where('user_id', $membership->user_id)
            ->whereNotIn('type', ['registration_fee'])
            ->latest('created_at')
            ->get()
            ->map(function($transaction) {
                return [
                    'date' => $transaction->created_at->format('Y-m-d H:i:s'),
                    'type' => $transaction->type_label,
                    'amount' => $transaction->amount,
                    'status' => ucfirst($transaction->status),
                    'description' => $transaction->description ?: 'Transaction processed',
                    'module' => 'Transactions'
                ];
            });

        // Combine all activities and sort by date
        $activities = $savingsTransactions
            ->concat($loanActivities)
            ->concat($registrationFeeTransactions)
            ->concat($otherTransactions)
            ->sortByDesc('date')
            ->values();

        $pdf = Pdf::loadView('adminlte.memberships.exports.activities_pdf', compact('membership', 'activities'))
                  ->setPaper('a4', 'portrait');

        return $pdf->download('member_' . $membership->membership_number . '_activities_' . date('Y-m-d_H-i-s') . '.pdf');
    }

    /**
     * Export memberships to CSV
     */
    public function exportCsv()
    {
        return Excel::download(new \App\Exports\MembershipsExport, 'memberships_report_' . date('Y-m-d_H-i-s') . '.csv');
    }

    /**
     * Generate and send login credentials for a member
     */
    public function generateCredentials(Membership $membership)
    {
        try {
            // Generate a random password
            $password = $this->generateRandomPassword();

            // Update the user's password
            $membership->user->update([
                'password' => bcrypt($password)
            ]);

            // Send login credentials email
            \Mail::to($membership->user->email)->send(new \App\Mail\MemberLoginCredentials($membership->user, $password));

            return redirect()->back()->with('success', 'Login credentials have been generated and sent to ' . $membership->user->email);

        } catch (\Exception $e) {
            \Log::error('Failed to generate credentials for member ' . $membership->id . ': ' . $e->getMessage());
            return redirect()->back()->with('error', 'Failed to generate credentials. Please try again.');
        }
    }

    /**
     * Generate random password
     */
    private function generateRandomPassword($length = 8)
    {
        $characters = '0123456789abcdefghijklmnopqrstuvwxyzABCDEFGHIJKLMNOPQRSTUVWXYZ!@#$%^&*';
        $password = '';

        for ($i = 0; $i < $length; $i++) {
            $password .= $characters[rand(0, strlen($characters) - 1)];
        }

        return $password;
    }

    /**
     * Bulk generate and send login credentials for multiple members
     */
    public function bulkGenerateCredentials(Request $request)
    {
        $request->validate([
            'member_ids' => 'required|string'
        ]);

        $memberIds = explode(',', $request->member_ids);
        $successCount = 0;
        $errors = [];

        foreach ($memberIds as $memberId) {
            try {
                $membership = Membership::findOrFail($memberId);

                // Generate a random password
                $password = $this->generateRandomPassword();

                // Update the user's password
                $membership->user->update([
                    'password' => bcrypt($password)
                ]);

                // Send login credentials email
                \Mail::to($membership->user->email)->send(new \App\Mail\MemberLoginCredentials($membership->user, $password));

                $successCount++;

            } catch (\Exception $e) {
                \Log::error('Failed to generate credentials for member ' . $memberId . ': ' . $e->getMessage());
                $errors[] = 'Failed for member ID ' . $memberId;
            }
        }

        $message = "Successfully generated credentials for {$successCount} member(s).";
        if (!empty($errors)) {
            $message .= " Errors: " . implode(', ', $errors);
        }

        return redirect()->back()->with('success', $message);
    }
}
