<?php

namespace App\Http\Controllers\Member;

use App\Http\Controllers\Controller;
use App\Models\Saving;
use Illuminate\Http\Request;

class SavingController extends Controller
{
    public function __construct()
    {
        $this->middleware('auth');
        $this->middleware('role:Member');
    }

    public function index()
    {
        $user = auth()->user();
        $savings = $user->savings()->latest()->paginate(10);
        $activeSavings = $user->savings()->where('status', 'active')->get();

        return view('member.savings.index', compact('savings', 'activeSavings'));
    }

    public function create()
    {
        return view('member.savings.create');
    }

    public function store(Request $request)
    {
        $user = auth()->user();

        $request->validate([
            'amount' => 'required|numeric|min:100',
            'type' => 'required|in:fixed,recurring',
            'term_months' => 'required_if:type,fixed|integer|min:1|max:120',
        ]);

        $data = [
            'user_id' => $user->id,
            'amount' => 0, // Initial amount 0, will be updated after payment
            'type' => $request->type,
            'status' => 'pending', // Pending until payment is confirmed
        ];

        if ($request->type === 'fixed') {
            $data['maturity_date'] = now()->addMonths((int) $request->term_months);
        }

        $saving = Saving::create($data);

        // Initiate M-Pesa payment for the initial deposit
        try {
            // Get M-Pesa credentials
            $mpesaConfig = \App\Models\MpesaCredential::where('is_active', true)->first();

            if (!$mpesaConfig) {
                return redirect()->route('member.savings.index')->with('error', 'M-Pesa configuration not found. Please contact administrator.');
            }

            // Initialize M-Pesa STK Push
            $mpesa = new \Safaricom\Mpesa\Mpesa();

            $response = $mpesa->stkpush([
                'BusinessShortCode' => $mpesaConfig->shortcode,
                'Password' => $this->generatePassword($mpesaConfig),
                'Timestamp' => date('YmdHis'),
                'TransactionType' => 'CustomerPayBillOnline',
                'Amount' => $request->amount,
                'PartyA' => $user->phone_number,
                'PartyB' => $mpesaConfig->shortcode,
                'PhoneNumber' => $user->phone_number,
                'CallBackURL' => route('mpesa.savings.deposit.callback'),
                'AccountReference' => 'SAVINGS-' . $saving->id,
                'TransactionDesc' => 'Initial Savings Deposit'
            ]);

            // Create transaction record
            \App\Models\Transaction::create([
                'user_id' => $user->id,
                'type' => 'savings_deposit',
                'amount' => $request->amount,
                'mpesa_transaction_id' => $response['MerchantRequestID'] ?? null,
                'status' => 'pending',
                'mpesa_response' => json_encode($response),
                'reference_id' => $saving->id,
                'notes' => 'Initial savings deposit initiated'
            ]);

            return redirect()->route('member.savings.index')->with('success', 'Savings account created! Please complete the M-Pesa payment to activate your account.');
        } catch (\Exception $e) {
            // If payment initiation fails, delete the created saving account
            $saving->delete();
            return redirect()->route('member.savings.index')->with('error', 'Failed to initiate payment: ' . $e->getMessage());
        }
    }

    public function show(Saving $saving)
    {
        // Ensure user can only view their own savings
        if ($saving->user_id !== auth()->id()) {
            abort(403);
        }

        return view('member.savings.show', compact('saving'));
    }

    public function deposit(Request $request, Saving $saving)
    {
        // Ensure user can only deposit to their own savings
        if ($saving->user_id !== auth()->id()) {
            abort(403);
        }

        $request->validate([
            'amount' => 'required|numeric|min:1'
        ]);

        try {
            // Get M-Pesa credentials
            $mpesaConfig = \App\Models\MpesaCredential::where('is_active', true)->first();

            if (!$mpesaConfig) {
                return response()->json([
                    'success' => false,
                    'message' => 'M-Pesa configuration not found. Please contact administrator.'
                ]);
            }

            // Initialize M-Pesa STK Push
            $mpesa = new \Safaricom\Mpesa\Mpesa();

            $response = $mpesa->stkpush([
                'BusinessShortCode' => $mpesaConfig->shortcode,
                'Password' => $this->generatePassword($mpesaConfig),
                'Timestamp' => date('YmdHis'),
                'TransactionType' => 'CustomerPayBillOnline',
                'Amount' => $request->amount,
                'PartyA' => $saving->user->phone_number,
                'PartyB' => $mpesaConfig->shortcode,
                'PhoneNumber' => $saving->user->phone_number,
                'CallBackURL' => route('mpesa.savings.deposit.callback'),
                'AccountReference' => 'SAVINGS-' . $saving->id,
                'TransactionDesc' => 'Savings Deposit'
            ]);

            // Create transaction record
            \App\Models\Transaction::create([
                'user_id' => $saving->user_id,
                'type' => 'savings_deposit',
                'amount' => $request->amount,
                'mpesa_transaction_id' => $response['MerchantRequestID'] ?? null,
                'status' => 'pending',
                'mpesa_response' => json_encode($response),
                'reference_id' => $saving->id,
                'notes' => 'Savings deposit initiated'
            ]);

            return response()->json([
                'success' => true,
                'message' => 'M-Pesa payment request sent successfully!',
                'data' => $response
            ]);

        } catch (\Exception $e) {
            return response()->json([
                'success' => false,
                'message' => 'Failed to initiate payment: ' . $e->getMessage()
            ]);
        }
    }

    public function withdraw(Request $request, Saving $saving)
    {
        // Ensure user can only withdraw from their own savings
        if ($saving->user_id !== auth()->id()) {
            abort(403);
        }

        $request->validate([
            'amount' => 'required|numeric|min:1|max:' . $saving->amount
        ]);

        // Update savings amount
        $saving->update([
            'amount' => $saving->amount - $request->amount,
            'status' => $saving->amount - $request->amount <= 0 ? 'withdrawn' : 'active'
        ]);

        // Create transaction record
        \App\Models\Transaction::create([
            'user_id' => $saving->user_id,
            'type' => 'savings_withdrawal',
            'amount' => $request->amount,
            'status' => 'completed',
            'reference_id' => $saving->id,
            'notes' => 'Savings withdrawal'
        ]);

        return redirect()->back()->with('success', 'Withdrawal processed successfully!');
    }

    private function generatePassword($mpesaConfig)
    {
        $timestamp = date('YmdHis');
        $password = base64_encode($mpesaConfig->shortcode . $mpesaConfig->passkey . $timestamp);
        return $password;
    }
}
