<?php

namespace App\Http\Controllers\Member;

use App\Http\Controllers\Controller;
use App\Models\Loan;
use Illuminate\Http\Request;
use Barryvdh\DomPDF\Facade\Pdf;

class LoanController extends Controller
{
    public function __construct()
    {
        $this->middleware('auth');
        $this->middleware('role:Member');
    }

    public function index()
    {
        $user = auth()->user();
        $loans = $user->loans()->latest()->paginate(10);

        return view('member.loans.index', compact('loans'));
    }

    public function create()
    {
        // Get default loan interest rate from system configuration
        $defaultInterestRate = \App\Models\SystemConfiguration::where('key', 'default_loan_interest_rate')->first()->value ?? 30;

        return view('member.loans.create', compact('defaultInterestRate'));
    }

    public function store(Request $request)
    {
        $user = auth()->user();

        $request->validate([
            'amount' => 'required|numeric|min:1000|max:500000',
            'application_date' => 'required|date|before_or_equal:today',
            'return_date' => 'required|date|after:application_date',
            'purpose' => 'required|string|max:500',
        ]);

        // Calculate term_months from dates
        $applicationDate = \Carbon\Carbon::parse($request->application_date);
        $returnDate = \Carbon\Carbon::parse($request->return_date);
        $termMonths = ceil($applicationDate->diffInMonths($returnDate));

        // Get default loan interest rate from system configuration
        $interestRate = \App\Models\SystemConfiguration::where('key', 'default_loan_interest_rate')->first()->value ?? 30;

        // Calculate total amount
        $totalAmount = $request->amount + ($request->amount * $interestRate / 100);
        $monthlyPayment = $termMonths > 0 ? $totalAmount / $termMonths : 0;

        // Create the loan
        $loan = Loan::create([
            'user_id' => $user->id,
            'amount' => $request->amount,
            'interest_rate' => $interestRate,
            'term_months' => $termMonths,
            'total_amount' => $totalAmount,
            'monthly_payment' => $monthlyPayment,
            'application_date' => $request->application_date,
            'return_date' => $request->return_date,
            'status' => 'pending',
            'notes' => $request->purpose,
        ]);


        return redirect()->route('member.loans.index')->with('success', 'Loan application submitted successfully!');
    }

    public function show(Loan $loan)
    {
        // Ensure user can only view their own loans
        if ($loan->user_id !== auth()->id()) {
            abort(403);
        }

        return view('member.loans.show', compact('loan'));
    }

    public function edit(Loan $loan)
    {
        // Ensure user can only edit their own loans
        if ($loan->user_id !== auth()->id()) {
            abort(403);
        }

        // Only allow editing if loan is still pending
        if ($loan->status !== 'pending') {
            return redirect()->route('member.loans.show', $loan)
                            ->with('error', 'You cannot edit an approved or processed loan application.');
        }

        // Get default loan interest rate from system configuration
        $defaultInterestRate = \App\Models\SystemConfiguration::where('key', 'default_loan_interest_rate')->first()->value ?? 30;

        return view('member.loans.edit', compact('loan', 'defaultInterestRate'));
    }

    public function update(Request $request, Loan $loan)
    {
        // Ensure user can only update their own loans
        if ($loan->user_id !== auth()->id()) {
            abort(403);
        }

        // Only allow updating if loan is still pending
        if ($loan->status !== 'pending') {
            return redirect()->route('member.loans.show', $loan)
                           ->with('error', 'You cannot update an approved or processed loan application.');
        }

        $request->validate([
            'amount' => 'required|numeric|min:1000|max:500000',
            'application_date' => 'required|date|before_or_equal:today',
            'return_date' => 'required|date|after:application_date',
            'purpose' => 'required|string|max:500',
        ]);

        // Calculate term_months from dates
        $applicationDate = \Carbon\Carbon::parse($request->application_date);
        $returnDate = \Carbon\Carbon::parse($request->return_date);
        $termMonths = ceil($applicationDate->diffInMonths($returnDate));

        // Get default loan interest rate from system configuration
        $interestRate = \App\Models\SystemConfiguration::where('key', 'default_loan_interest_rate')->first()->value ?? 30;

        // Calculate total amount
        $totalAmount = $request->amount + ($request->amount * $interestRate / 100);
        $monthlyPayment = $termMonths > 0 ? $totalAmount / $termMonths : 0;

        // Update the loan
        $loan->update([
            'amount' => $request->amount,
            'interest_rate' => $interestRate,
            'term_months' => $termMonths,
            'total_amount' => $totalAmount,
            'monthly_payment' => $monthlyPayment,
            'application_date' => $request->application_date,
            'return_date' => $request->return_date,
            'notes' => $request->purpose,
        ]);


        return redirect()->route('member.loans.show', $loan)->with('success', 'Loan application updated successfully!');
    }

    public function destroy(Loan $loan)
    {
        // Ensure user can only delete their own loans
        if ($loan->user_id !== auth()->id()) {
            abort(403);
        }

        // Only allow deleting if loan is still pending
        if ($loan->status !== 'pending') {
            return redirect()->route('member.loans.show', $loan)
                           ->with('error', 'You cannot delete an approved or processed loan application.');
        }

        $loan->delete();

        return redirect()->route('member.loans.index')->with('success', 'Loan application deleted successfully!');
    }

    public function payLoan(Request $request, Loan $loan)
    {
        // Ensure user can only pay their own loans
        if ($loan->user_id !== auth()->id()) {
            abort(403);
        }

        $request->validate([
            'amount' => 'required|numeric|min:1'
        ]);

        try {
            // Get M-Pesa credentials
            $mpesaConfig = \App\Models\MpesaCredential::where('is_active', true)->first();

            if (!$mpesaConfig) {
                return response()->json([
                    'success' => false,
                    'message' => 'M-Pesa configuration not found. Please contact administrator.'
                ]);
            }

            // Initialize M-Pesa STK Push
            $mpesa = new \Safaricom\Mpesa\Mpesa();

            $response = $mpesa->stkpush([
                'BusinessShortCode' => $mpesaConfig->shortcode,
                'Password' => $this->generatePassword($mpesaConfig),
                'Timestamp' => date('YmdHis'),
                'TransactionType' => 'CustomerPayBillOnline',
                'Amount' => $request->amount,
                'PartyA' => $loan->user->phone_number,
                'PartyB' => $mpesaConfig->shortcode,
                'PhoneNumber' => $loan->user->phone_number,
                'CallBackURL' => route('mpesa.loan.payment.callback'),
                'AccountReference' => 'LOAN-PAY-' . $loan->id,
                'TransactionDesc' => 'Loan Payment'
            ]);

            // Create transaction record
            \App\Models\Transaction::create([
                'user_id' => $loan->user_id,
                'type' => 'loan_repayment',
                'amount' => $request->amount,
                'mpesa_transaction_id' => $response['MerchantRequestID'] ?? null,
                'status' => 'pending',
                'mpesa_response' => json_encode($response),
                'reference_id' => $loan->id,
                'notes' => 'Loan payment initiated'
            ]);

            return response()->json([
                'success' => true,
                'message' => 'M-Pesa payment request sent successfully!',
                'data' => $response
            ]);

        } catch (\Exception $e) {
            return response()->json([
                'success' => false,
                'message' => 'Failed to initiate payment: ' . $e->getMessage()
            ]);
        }
    }

    /**
     * Export member loans to PDF.
     */
    public function exportPdf()
    {
        $user = auth()->user();
        $loans = $user->loans()->with('user')->latest()->get();

        $pdf = Pdf::loadView('member.loans.exports.pdf', compact('loans'))
                  ->setPaper('a4', 'landscape');

        $filename = 'my_loans_' . date('Y-m-d_H-i-s') . '.pdf';
        return $pdf->download($filename);
    }


    private function generatePassword($mpesaConfig)
    {
        $timestamp = date('YmdHis');
        $password = base64_encode($mpesaConfig->shortcode . $mpesaConfig->passkey . $timestamp);
        return $password;
    }
}
