<?php

namespace App\Http\Controllers\Member;

use App\Http\Controllers\Controller;
use App\Models\Loan;
use App\Models\Saving;
use App\Models\Share;
use App\Models\Transaction;
use App\Models\MoneyMarketAccount;
use Illuminate\Http\Request;

class DashboardController extends Controller
{
    public function __construct()
    {
        $this->middleware('auth');
        $this->middleware('role:Member');
    }

    public function index()
    {
        $user = auth()->user();

        // Get user's loans
        $loans = $user->loans()->latest()->take(5)->get();

        // Get user's savings
        $savings = $user->savings()->latest()->take(5)->get();

        // Get user's shares
        $shares = $user->shares()->latest()->take(5)->get();

        // Get user's money market accounts
        $moneyMarketAccounts = MoneyMarketAccount::where('membership_id', $user->membership->id ?? null)
            ->latest()
            ->take(5)
            ->get();

        // Get recent transactions
        $transactions = $user->transactions()->latest()->take(10)->get();

        // Calculate totals
        $totalLoans = $user->loans()->whereIn('status', ['active', 'approved'])->sum('amount');
        $totalSavings = $user->savings()->where('status', 'active')->sum('amount');
        $totalShares = $user->shares()->sum('total_value');
        $totalSharesCount = $user->shares()->sum('number_of_shares');
        $totalMoneyMarket = MoneyMarketAccount::where('membership_id', $user->membership->id ?? null)
            ->whereIn('status', ['active', 'matured'])
            ->sum('total_amount');
        $pendingLoans = $user->loans()->where('status', 'pending')->count();

        // Calculate loan repayment progress
        $loanProgress = [];
        foreach ($loans as $loan) {
            $totalPaid = $user->transactions()
                ->where('type', 'loan_payment')
                ->where('reference_id', $loan->id)
                ->sum('amount');
            $progress = $loan->amount > 0 ? min(($totalPaid / $loan->amount) * 100, 100) : 0;
            $loanProgress[$loan->id] = [
                'paid' => $totalPaid,
                'progress' => $progress
            ];
        }

        // Get monthly savings trend (last 6 months)
        $savingsTrend = [];
        for ($i = 5; $i >= 0; $i--) {
            $date = now()->subMonths($i);
            $monthSavings = $user->transactions()
                ->where('type', 'savings_deposit')
                ->whereYear('created_at', $date->year)
                ->whereMonth('created_at', $date->month)
                ->sum('amount');
            $savingsTrend[] = [
                'month' => $date->format('M Y'),
                'amount' => $monthSavings
            ];
        }

        // Get monthly loan payments trend (last 6 months)
        $loansTrend = [];
        for ($i = 5; $i >= 0; $i--) {
            $date = now()->subMonths($i);
            $monthLoans = $user->transactions()
                ->where('type', 'loan_payment')
                ->whereYear('created_at', $date->year)
                ->whereMonth('created_at', $date->month)
                ->sum('amount');
            $loansTrend[] = [
                'month' => $date->format('M Y'),
                'amount' => $monthLoans
            ];
        }

        // Calculate savings growth (compare current month to previous)
        $currentMonthSavings = $savingsTrend[5]['amount'] ?? 0;
        $previousMonthSavings = $savingsTrend[4]['amount'] ?? 0;
        $savingsGrowth = $previousMonthSavings > 0 ? (($currentMonthSavings - $previousMonthSavings) / $previousMonthSavings) * 100 : 0;

        // Get active loans count
        $activeLoans = $user->loans()->whereIn('status', ['active', 'approved'])->count();

        return view('member.dashboard', compact(
            'loans',
            'savings',
            'shares',
            'moneyMarketAccounts',
            'transactions',
            'totalLoans',
            'totalSavings',
            'totalShares',
            'totalSharesCount',
            'totalMoneyMarket',
            'pendingLoans',
            'loanProgress',
            'savingsTrend',
            'loansTrend',
            'savingsGrowth',
            'activeLoans'
        ));
    }

    public function profile()
    {
        $user = auth()->user()->load('membership', 'branch');
        return view('member.profile', compact('user'));
    }

    public function updateProfile(Request $request)
    {
        $user = auth()->user();

        $request->validate([
            'first_name' => 'required|string|max:255',
            'second_name' => 'required|string|max:255',
            'phone_number' => 'required|string|unique:users,phone_number,' . $user->id,
            'email' => 'required|email|unique:users,email,' . $user->id,
        ]);

        $user->update([
            'name' => $request->first_name . ' ' . $request->second_name,
            'first_name' => $request->first_name,
            'second_name' => $request->second_name,
            'phone_number' => $request->phone_number,
            'email' => $request->email,
        ]);

        return redirect()->back()->with('success', 'Profile updated successfully!');
    }
}
