<?php

namespace App\Http\Controllers;

use Illuminate\Http\Request;
use App\Models\Membership;
use App\Models\Loan;
use App\Models\Saving;
use App\Models\Transaction;
use App\Models\Branch;
use App\Models\Share;
use App\Models\Dividend;

class DashboardController extends Controller
{
    public function index()
    {
        $user = auth()->user();

        // Redirect based on user role
        if ($user->hasRole('Admin')) {
            return $this->adminDashboard();
        } elseif ($user->hasRole('Loan Manager')) {
            return $this->loanManagerDashboard();
        } elseif ($user->hasRole('Front Office')) {
            return $this->frontOfficeDashboard();
        } elseif ($user->hasRole('Member')) {
            return $this->memberDashboard();
        }

        // Default fallback
        return view('adminlte::page', [
            'content' => 'Welcome to Morio Sacco',
            'title' => 'Dashboard'
        ]);
    }

    public function futuristicDashboard()
    {
        $user = auth()->user();

        // Only allow admins to access the futuristic dashboard
        if (!$user->hasRole('Admin')) {
            return redirect()->route('home');
        }

        // Comprehensive real-time data for futuristic admin dashboard
        $stats = $this->getComprehensiveDashboardData();

        return view('dashboards.admin', compact('stats'));
    }

    private function adminDashboard()
    {
        // Comprehensive real-time data for admin dashboard
        $stats = $this->getComprehensiveDashboardData();

        return view('adminlte::page', [
            'content' => view('dashboards.admin', compact('stats')),
            'title' => 'Admin Dashboard'
        ]);
    }

    private function getComprehensiveDashboardData()
    {
        try {
            // Real-time membership data
            $totalMembers = Membership::count();
            $activeMembers = Membership::where('status', 'active')->count();
            $pendingMembers = Membership::where('status', 'pending')->count();
            $newMembersThisMonth = Membership::whereMonth('created_at', date('m'))->count();

            // Real-time loan data
            $totalLoans = Loan::count();
            $activeLoans = Loan::where('status', 'active')->count();
            $pendingLoans = Loan::where('status', 'pending')->count();
            $approvedLoans = Loan::where('status', 'approved')->count();
            $totalLoanAmount = Loan::whereIn('status', ['active', 'approved'])->sum('amount') ?? 0;
            $loansThisMonth = Loan::whereMonth('created_at', date('m'))->count();

            // Real-time savings data
            $totalSavingsAccounts = Saving::count();
            $activeSavingsAccounts = Saving::where('status', 'active')->count();
            $totalSavingsBalance = Saving::where('status', 'active')->sum('amount') ?? 0;

            // Handle potential issues with SavingsTransaction
            try {
                $savingsDepositsThisMonth = \App\Models\SavingsTransaction::where('type', 'deposit')
                    ->whereMonth('transaction_date', date('m'))
                    ->sum('amount') ?? 0;
            } catch (\Exception $e) {
                $savingsDepositsThisMonth = 0;
            }

            // Real-time transaction data
            $totalTransactions = Transaction::count();
            $completedTransactions = Transaction::where('status', 'completed')->count();
            $pendingTransactions = Transaction::where('status', 'pending')->count();
            $monthlyRevenue = Transaction::where('status', 'completed')
                ->whereMonth('created_at', date('m'))
                ->sum('amount') ?? 0;

            // Branch data
            $totalBranches = Branch::count();
            $activeBranches = Branch::count(); // All branches are considered active since there's no status column

            // Recent activities (last 24 hours)
            $recentActivities = $this->getRecentSystemActivities();

            // Monthly trends for charts
            $monthlyTrends = $this->getMonthlyDashboardTrends();

            // System health metrics
            $systemHealth = $this->getSystemHealthMetrics();

            // Top performing branches
            $topBranches = $this->getTopPerformingBranches();

            // Loan performance metrics
            $loanPerformance = $this->getLoanPerformanceMetrics();

            // Top borrower, top saver, top shareholder, top defaulter, and defaulted loans
            $topBorrower = $this->getTopBorrower();
            $topSaver = $this->getTopSaver();
            $topShareholder = $this->getTopShareholder();
            $topDefaulter = $this->getTopDefaulter();
            $defaultedLoans = $this->getDefaultedLoans();

            return [
                // Membership Metrics
                'total_members' => $totalMembers,
                'active_members' => $activeMembers,
                'pending_members' => $pendingMembers,
                'new_members_this_month' => $newMembersThisMonth,

                // Loan Metrics
                'total_loans' => $totalLoans,
                'active_loans' => $activeLoans,
                'pending_loans' => $pendingLoans,
                'approved_loans' => $approvedLoans,
                'total_loan_amount' => $totalLoanAmount,
                'loans_this_month' => $loansThisMonth,

                // Savings Metrics
                'total_savings_accounts' => $totalSavingsAccounts,
                'active_savings_accounts' => $activeSavingsAccounts,
                'total_savings_balance' => $totalSavingsBalance,
                'savings_deposits_this_month' => $savingsDepositsThisMonth,

                // Transaction Metrics
                'total_transactions' => $totalTransactions,
                'completed_transactions' => $completedTransactions,
                'pending_transactions' => $pendingTransactions,
                'monthly_revenue' => $monthlyRevenue,

                // Branch Metrics
                'total_branches' => $totalBranches,
                'active_branches' => $activeBranches,

                // Dynamic Data
                'recent_activities' => $recentActivities,
                'monthly_trends' => $monthlyTrends,
                'system_health' => $systemHealth,
                'top_branches' => $topBranches,
                'loan_performance' => $loanPerformance,
                'top_borrower' => $topBorrower,
                'top_saver' => $topSaver,
                'top_shareholder' => $topShareholder,
                'top_defaulter' => $topDefaulter,
                'defaulted_loans' => $defaultedLoans,

                // Legacy data for backward compatibility
                'recent_transactions' => Transaction::latest()->take(5)->get(),
            ];
        } catch (\Exception $e) {
            // Return default data if there's a major error
            \Log::error('Dashboard data error: ' . $e->getMessage());
            return [
                'total_members' => 0,
                'active_members' => 0,
                'pending_members' => 0,
                'new_members_this_month' => 0,
                'total_loans' => 0,
                'active_loans' => 0,
                'pending_loans' => 0,
                'approved_loans' => 0,
                'total_loan_amount' => 0,
                'loans_this_month' => 0,
                'total_savings_accounts' => 0,
                'active_savings_accounts' => 0,
                'total_savings_balance' => 0,
                'savings_deposits_this_month' => 0,
                'total_transactions' => 0,
                'completed_transactions' => 0,
                'pending_transactions' => 0,
                'monthly_revenue' => 0,
                'total_branches' => 0,
                'active_branches' => 0,
                'recent_activities' => collect(),
                'monthly_trends' => [['month' => 'No Data', 'memberships' => 0, 'loans' => 0, 'transactions' => 0, 'savings' => 0]],
                'system_health' => ['transaction_success_rate' => 0, 'loan_utilization_rate' => 0, 'active_members_percentage' => 0, 'system_uptime' => 'N/A', 'last_backup' => 'N/A'],
                'top_branches' => collect([['name' => 'No data available', 'members' => 0, 'savings' => 0, 'loans' => 0, 'performance_score' => 0]]),
                'loan_performance' => ['total_loan_portfolio' => 0, 'total_repayments' => 0, 'repayment_rate' => 0, 'on_time_payment_rate' => 0, 'average_loan_size' => 0],
                'recent_transactions' => collect(),
            ];
        }
    }

    private function getRecentSystemActivities()
    {
        $activities = collect();

        // Recent memberships
        $memberships = Membership::with('user')
            ->where('created_at', '>=', now()->subHours(24))
            ->orderBy('created_at', 'desc')
            ->take(3)
            ->get()
            ->map(function ($membership) {
                return [
                    'type' => 'membership',
                    'action' => 'New member registered',
                    'details' => $membership->user->full_name,
                    'amount' => $membership->share_capital,
                    'timestamp' => $membership->created_at,
                    'icon' => 'fas fa-user-plus text-primary',
                    'color' => 'primary'
                ];
            });

        // Recent loans
        $loans = Loan::with('user')
            ->where('created_at', '>=', now()->subHours(24))
            ->orderBy('created_at', 'desc')
            ->take(3)
            ->get()
            ->map(function ($loan) {
                return [
                    'type' => 'loan',
                    'action' => 'Loan application submitted',
                    'details' => $loan->user->full_name,
                    'amount' => $loan->amount,
                    'timestamp' => $loan->created_at,
                    'icon' => 'fas fa-money-bill-wave text-warning',
                    'color' => 'warning'
                ];
            });

        // Recent transactions
        $transactions = Transaction::with('user')
            ->where('created_at', '>=', now()->subHours(24))
            ->orderBy('created_at', 'desc')
            ->take(3)
            ->get()
            ->map(function ($transaction) {
                return [
                    'type' => 'transaction',
                    'action' => ucfirst(str_replace('_', ' ', $transaction->type)),
                    'details' => $transaction->user->full_name,
                    'amount' => $transaction->amount,
                    'timestamp' => $transaction->created_at,
                    'icon' => 'fas fa-exchange-alt text-success',
                    'color' => 'success'
                ];
            });

        return $activities->concat($memberships)->concat($loans)->concat($transactions)
            ->sortByDesc('timestamp')->take(10)->values();
    }

    private function getMonthlyDashboardTrends()
    {
        try {
            $currentMonth = now();
            $monthlyData = [];

            // Last 6 months data
            for ($i = 5; $i >= 0; $i--) {
                $date = $currentMonth->copy()->subMonths($i);
                $monthName = $date->format('M Y');

                $memberships = Membership::whereYear('created_at', $date->year)
                    ->whereMonth('created_at', $date->month)->count();

                $loans = Loan::whereYear('created_at', $date->year)
                    ->whereMonth('created_at', $date->month)->count();

                $transactions = Transaction::where('status', 'completed')
                    ->whereYear('created_at', $date->year)
                    ->whereMonth('created_at', $date->month)
                    ->sum('amount') ?? 0;

                // Handle potential issues with SavingsTransaction
                try {
                    $savings = \App\Models\SavingsTransaction::where('type', 'deposit')
                        ->whereYear('transaction_date', $date->year)
                        ->whereMonth('transaction_date', $date->month)
                        ->sum('amount') ?? 0;
                } catch (\Exception $e) {
                    $savings = 0;
                }

                $monthlyData[] = [
                    'month' => $monthName,
                    'memberships' => $memberships,
                    'loans' => $loans,
                    'transactions' => $transactions,
                    'savings' => $savings
                ];
            }

            return $monthlyData;
        } catch (\Exception $e) {
            // Return default data if there's an error
            return [
                ['month' => 'No Data', 'memberships' => 0, 'loans' => 0, 'transactions' => 0, 'savings' => 0]
            ];
        }
    }

    private function getSystemHealthMetrics()
    {
        $totalTransactions = Transaction::count();
        $completedTransactions = Transaction::where('status', 'completed')->count();
        $successRate = $totalTransactions > 0 ? ($completedTransactions / $totalTransactions) * 100 : 0;

        $activeLoans = Loan::where('status', 'active')->count();
        $totalLoans = Loan::count();
        $loanUtilization = $totalLoans > 0 ? ($activeLoans / $totalLoans) * 100 : 0;

        return [
            'transaction_success_rate' => round($successRate, 1),
            'loan_utilization_rate' => round($loanUtilization, 1),
            'active_members_percentage' => Membership::count() > 0 ?
                round((Membership::where('status', 'active')->count() / Membership::count()) * 100, 1) : 0,
            'system_uptime' => '99.9%', // Placeholder - could be calculated from logs
            'last_backup' => now()->subHours(6)->diffForHumans(), // Placeholder
        ];
    }

    private function getTopPerformingBranches()
    {
        try {
            return Branch::withCount(['users as member_count' => function ($query) {
                $query->whereHas('membership', function ($q) {
                    $q->where('status', 'active');
                });
            }])
            ->with(['users' => function ($query) {
                $query->with('membership')->whereHas('membership', function ($q) {
                    $q->where('status', 'active');
                });
            }])
            ->get()
            ->map(function ($branch) {
                try {
                    $totalSavings = $branch->users->sum(function ($user) {
                        return $user->savings()->where('status', 'active')->sum('amount');
                    });

                    $totalLoans = $branch->users->sum(function ($user) {
                        return $user->loans()->whereIn('status', ['active', 'approved'])->sum('amount');
                    });

                    return [
                        'name' => $branch->name,
                        'members' => $branch->member_count,
                        'savings' => $totalSavings,
                        'loans' => $totalLoans,
                        'performance_score' => $branch->member_count + ($totalSavings / 10000) + ($totalLoans / 50000)
                    ];
                } catch (\Exception $e) {
                    // Return default values if there's an error
                    return [
                        'name' => $branch->name,
                        'members' => $branch->member_count ?? 0,
                        'savings' => 0,
                        'loans' => 0,
                        'performance_score' => $branch->member_count ?? 0
                    ];
                }
            })
            ->sortByDesc('performance_score')
            ->take(5)
            ->values();
        } catch (\Exception $e) {
            // Return empty array if there's a major error
            return collect([
                ['name' => 'No data available', 'members' => 0, 'savings' => 0, 'loans' => 0, 'performance_score' => 0]
            ]);
        }
    }

    private function getLoanPerformanceMetrics()
    {
        $totalLoanAmount = Loan::whereIn('status', ['active', 'approved'])->sum('amount');
        $totalRepayments = Transaction::where('type', 'loan_repayment')
            ->where('status', 'completed')
            ->sum('amount');

        $onTimePayments = Transaction::where('type', 'loan_repayment')
            ->where('status', 'completed')
            ->whereRaw('DATEDIFF(created_at, DATE_SUB(created_at, INTERVAL 1 MONTH)) <= 30')
            ->count();

        $totalPayments = Transaction::where('type', 'loan_repayment')
            ->where('status', 'completed')
            ->count();

        $onTimeRate = $totalPayments > 0 ? ($onTimePayments / $totalPayments) * 100 : 0;

        return [
            'total_loan_portfolio' => $totalLoanAmount,
            'total_repayments' => $totalRepayments,
            'repayment_rate' => $totalLoanAmount > 0 ? round(($totalRepayments / $totalLoanAmount) * 100, 1) : 0,
            'on_time_payment_rate' => round($onTimeRate, 1),
            'average_loan_size' => Loan::whereIn('status', ['active', 'approved'])->avg('amount') ?? 0,
        ];
    }

    private function getTopBorrower()
    {
        try {
            $topBorrower = Loan::with('user')
                ->whereIn('status', ['active', 'approved'])
                ->orderBy('amount', 'desc')
                ->first();

            if ($topBorrower) {
                return [
                    'name' => $topBorrower->user->full_name ?? 'N/A',
                    'amount' => $topBorrower->amount,
                    'loan_type' => $topBorrower->type ?? 'N/A',
                ];
            }

            return [
                'name' => 'No active loans',
                'amount' => 0,
                'loan_type' => 'N/A',
            ];
        } catch (\Exception $e) {
            return [
                'name' => 'Data unavailable',
                'amount' => 0,
                'loan_type' => 'N/A',
            ];
        }
    }

    private function getTopSaver()
    {
        try {
            // Find the user with the highest total savings balance
            $topSaver = \App\Models\User::with(['savings' => function($query) {
                $query->where('status', 'active');
            }])
            ->whereHas('savings', function($query) {
                $query->where('status', 'active');
            })
            ->get()
            ->map(function($user) {
                $totalSavings = $user->savings->sum('amount');
                $user->total_savings = $totalSavings;
                return $user;
            })
            ->sortByDesc('total_savings')
            ->first();

            if ($topSaver && $topSaver->total_savings > 0) {
                return [
                    'name' => $topSaver->full_name ?? 'N/A',
                    'amount' => $topSaver->total_savings,
                    'accounts' => $topSaver->savings->count(),
                ];
            }

            return [
                'name' => 'No active savings',
                'amount' => 0,
                'accounts' => 0,
            ];
        } catch (\Exception $e) {
            return [
                'name' => 'Data unavailable',
                'amount' => 0,
                'accounts' => 0,
            ];
        }
    }

    private function getTopShareholder()
    {
        try {
            // Find the user with the highest total share value
            $topShareholder = \App\Models\User::with(['shares'])
                ->whereHas('shares')
                ->get()
                ->map(function($user) {
                    $totalSharesValue = $user->shares->sum('total_value');
                    $user->total_shares_value = $totalSharesValue;
                    return $user;
                })
                ->sortByDesc('total_shares_value')
                ->first();

            if ($topShareholder && $topShareholder->total_shares_value > 0) {
                return [
                    'name' => $topShareholder->full_name ?? 'N/A',
                    'amount' => $topShareholder->total_shares_value,
                    'shares_count' => $topShareholder->shares->sum('number_of_shares'),
                ];
            }

            return [
                'name' => 'No shareholders',
                'amount' => 0,
                'shares_count' => 0,
            ];
        } catch (\Exception $e) {
            return [
                'name' => 'Data unavailable',
                'amount' => 0,
                'shares_count' => 0,
            ];
        }
    }

    private function getDefaultedLoans()
    {
        try {
            // Count loans that are past due (assuming loans have due dates)
            // For now, we'll count loans with status 'defaulted' or similar
            // You may need to adjust this based on your loan status definitions
            $defaultedLoans = Loan::where('status', 'defaulted')->count();
            $defaultedAmount = Loan::where('status', 'defaulted')->sum('amount');

            // Alternative: loans that are overdue based on due_date
            $overdueLoans = Loan::where('status', 'active')
                ->where('due_date', '<', now())
                ->count();

            $overdueAmount = Loan::where('status', 'active')
                ->where('due_date', '<', now())
                ->sum('amount');

            return [
                'count' => $defaultedLoans + $overdueLoans,
                'amount' => $defaultedAmount + $overdueAmount,
                'defaulted_count' => $defaultedLoans,
                'overdue_count' => $overdueLoans,
            ];
        } catch (\Exception $e) {
            return [
                'count' => 0,
                'amount' => 0,
                'defaulted_count' => 0,
                'overdue_count' => 0,
            ];
        }
    }

    private function getTopDefaulter()
    {
        try {
            // Find the user with the highest defaulted/overdue loan amount
            $topDefaulter = \App\Models\User::with(['loans' => function($query) {
                $query->where(function($q) {
                    $q->where('status', 'defaulted')
                      ->orWhere(function($subQuery) {
                          $subQuery->where('status', 'active')
                                   ->where('due_date', '<', now());
                      });
                });
            }])
            ->whereHas('loans', function($query) {
                $query->where(function($q) {
                    $q->where('status', 'defaulted')
                      ->orWhere(function($subQuery) {
                          $subQuery->where('status', 'active')
                                   ->where('due_date', '<', now());
                      });
                });
            })
            ->get()
            ->map(function($user) {
                $defaultedAmount = $user->loans->sum('amount');
                $user->total_defaulted_amount = $defaultedAmount;
                return $user;
            })
            ->sortByDesc('total_defaulted_amount')
            ->first();

            if ($topDefaulter && $topDefaulter->total_defaulted_amount > 0) {
                return [
                    'name' => $topDefaulter->full_name ?? 'N/A',
                    'amount' => $topDefaulter->total_defaulted_amount,
                    'loan_count' => $topDefaulter->loans->count(),
                ];
            }

            return [
                'name' => 'No defaulters',
                'amount' => 0,
                'loan_count' => 0,
            ];
        } catch (\Exception $e) {
            return [
                'name' => 'Data unavailable',
                'amount' => 0,
                'loan_count' => 0,
            ];
        }
    }

    private function loanManagerDashboard()
    {
        // Loan Manager focuses on loans and savings
        $stats = [
            'pending_loans' => Loan::where('status', 'pending')->count(),
            'approved_loans' => Loan::where('status', 'approved')->count(),
            'active_loans' => Loan::where('status', 'active')->count(),
            'total_loan_amount' => Loan::whereIn('status', ['approved', 'active'])->sum('amount'),
            'total_savings' => Saving::sum('amount'),
            'recent_loans' => Loan::with('user')->latest()->take(10)->get(),
            'loan_approvals_needed' => Loan::where('status', 'pending')->count(),
        ];

        return view('adminlte::page', [
            'content' => view('dashboards.loan_manager', compact('stats')),
            'title' => 'Loan Manager Dashboard'
        ]);
    }

    private function frontOfficeDashboard()
    {
        // Front Office handles member services
        $stats = [
            'pending_memberships' => Membership::where('status', 'pending')->count(),
            'total_members' => Membership::where('status', 'active')->count(),
            'recent_applications' => Membership::latest()->take(5)->get(),
            'branch_members' => Membership::whereHas('user', function($q) {
                $q->where('branch_id', auth()->user()->branch_id);
            })->count(),
            'pending_transactions' => Transaction::where('status', 'pending')->count(),
        ];

        return view('adminlte::page', [
            'content' => view('dashboards.front_office', compact('stats')),
            'title' => 'Front Office Dashboard'
        ]);
    }

    private function memberDashboard()
    {
        // Redirect members to their dedicated dashboard
        return redirect()->route('member.dashboard');
    }

    /**
     * Get real-time dashboard data via AJAX
     */
    public function getRealTimeDashboardData(Request $request)
    {
        try {
            $data = $this->getComprehensiveDashboardData();

            // Add timestamp for client-side tracking
            $data['timestamp'] = now()->toISOString();
            $data['last_updated'] = now()->diffForHumans();

            return response()->json([
                'success' => true,
                'data' => $data
            ]);
        } catch (\Exception $e) {
            return response()->json([
                'success' => false,
                'message' => 'Failed to fetch real-time data',
                'error' => $e->getMessage()
            ], 500);
        }
    }
}
