<?php

namespace App\Http\Controllers;

use Illuminate\Http\Request;
use App\Models\Branch;
use Maatwebsite\Excel\Facades\Excel;
use Barryvdh\DomPDF\Facade\Pdf;
use Illuminate\Support\Facades\Storage;

class BranchController extends Controller
{
    /**
     * Display a listing of the resource.
     */
    public function index()
    {
        $branches = Branch::withCount('users')->paginate(10);

        return view('adminlte.branches.index', compact('branches'));
    }

    /**
     * Show the form for creating a new resource.
     */
    public function create()
    {
        return view('adminlte.branches.create');
    }

    /**
     * Store a newly created resource in storage.
     */
    public function store(Request $request)
    {
        $request->validate([
            'name' => 'required|string|max:255',
            'location' => 'required|string|max:255',
            'contact' => 'nullable|string|max:255',
            'manager' => 'nullable|string|max:255',
            'description' => 'nullable|string',
        ]);

        Branch::create($request->all());

        return redirect()->route('branches.index')
                        ->with('success', 'Branch created successfully!');
    }

    /**
     * Display the specified resource.
     */
    public function show(Branch $branch)
    {
        $branch->load('users.membership');
        return view('adminlte.branches.show', compact('branch'));
    }

    /**
     * Show the form for editing the specified resource.
     */
    public function edit(Branch $branch)
    {
        return view('adminlte.branches.edit', compact('branch'));
    }

    /**
     * Update the specified resource in storage.
     */
    public function update(Request $request, Branch $branch)
    {
        $request->validate([
            'name' => 'required|string|max:255',
            'location' => 'required|string|max:255',
            'contact' => 'nullable|string|max:255',
            'manager' => 'nullable|string|max:255',
            'description' => 'nullable|string',
        ]);

        $branch->update($request->all());

        return redirect()->route('branches.index')
                        ->with('success', 'Branch updated successfully!');
    }

    /**
     * Remove the specified resource from storage.
     */
    public function destroy(Branch $branch)
    {
        // Check if branch has members
        if ($branch->users()->count() > 0) {
            return redirect()->route('branches.index')
                            ->with('error', 'Cannot delete branch with existing members. Please reassign members first.');
        }

        $branch->delete();

        return redirect()->route('branches.index')
                        ->with('success', 'Branch deleted successfully!');
    }

    /**
     * Export branches to PDF
     */
    public function exportPdf()
    {
        $branches = Branch::withCount('users')->get();
        $appUrl = config('app.url');

        $pdf = Pdf::loadView('adminlte.branches.exports.pdf', compact('branches', 'appUrl'));

        return $pdf->download('branches_' . date('Y-m-d_H-i-s') . '.pdf');
    }

    /**
     * Export branches to Excel
     */
    public function exportExcel()
    {
        return Excel::download(new \App\Exports\BranchesExport, 'branches_' . date('Y-m-d_H-i-s') . '.xlsx');
    }

    /**
     * Export branches to CSV
     */
    public function exportCsv()
    {
        return Excel::download(new \App\Exports\BranchesExport, 'branches_' . date('Y-m-d_H-i-s') . '.csv', \Maatwebsite\Excel\Excel::CSV);
    }
}
